/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.sidecar.tasks;

import io.vertx.core.Closeable;
import io.vertx.core.Future;
import io.vertx.core.Handler;
import io.vertx.core.Promise;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.cassandra.sidecar.common.server.utils.DurationSpec;
import org.apache.cassandra.sidecar.concurrent.ExecutorPools;
import org.apache.cassandra.sidecar.concurrent.TaskExecutorPool;
import org.apache.cassandra.sidecar.coordination.ClusterLease;
import org.apache.cassandra.sidecar.coordination.ExecuteOnClusterLeaseholderOnly;
import org.apache.cassandra.sidecar.tasks.PeriodicTask;
import org.apache.cassandra.sidecar.tasks.ScheduleDecision;
import org.jetbrains.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PeriodicTaskExecutor
implements Closeable {
    private static final Logger LOGGER = LoggerFactory.getLogger(PeriodicTaskExecutor.class);
    private static final long RUN_NOW_TIMER_ID = -1L;
    private static final long UNSCHEDULED_STATE_TIMER_ID = -2L;
    private final Map<PeriodicTaskKey, Long> timerIds = new ConcurrentHashMap<PeriodicTaskKey, Long>();
    private final Map<PeriodicTaskKey, Future<Void>> activeRuns = new ConcurrentHashMap<PeriodicTaskKey, Future<Void>>();
    private final TaskExecutorPool internalPool;
    private final ClusterLease clusterLease;

    public PeriodicTaskExecutor(ExecutorPools executorPools, ClusterLease clusterLease) {
        this.internalPool = executorPools.internal();
        this.clusterLease = clusterLease;
    }

    public void schedule(PeriodicTask task) {
        PeriodicTaskKey key = new PeriodicTaskKey(task);
        this.schedule(key, 0L, task.initialDelay().to(TimeUnit.MILLISECONDS), 0L);
    }

    private void schedule(PeriodicTaskKey key, long priorExecDurationMillis, long delayMillis, long execCount) {
        long actualDelayMillis = delayMillis - priorExecDurationMillis;
        AtomicBoolean runNow = new AtomicBoolean(actualDelayMillis <= 0L);
        this.timerIds.compute(key, (k, tid) -> {
            if (tid != null && execCount == 0L) {
                LOGGER.debug("Task is already scheduled. task='{}'", (Object)key);
                runNow.set(false);
                return tid;
            }
            if (tid != null && tid == -2L) {
                LOGGER.debug("Task is now unscheduled. task='{}' execCount={}", (Object)key, (Object)execCount);
                runNow.set(false);
                return null;
            }
            if (tid == null && execCount != 0L) {
                LOGGER.info("The executor is closed or the task is already unscheduled. Avoid scheduling more runs.tid=null task='{}' execCount={}", (Object)key, (Object)execCount);
                runNow.set(false);
                return null;
            }
            LOGGER.debug("Scheduling task {}. task='{}' execCount={}", new Object[]{runNow.get() ? "immediately" : "in " + actualDelayMillis + " milliseconds", key, execCount});
            try {
                key.task.registerPeriodicTaskExecutor(this);
            }
            catch (Exception e) {
                LOGGER.warn("Failed to invoke registerPeriodicTaskExecutor. task='{}'", (Object)key, (Object)e);
            }
            if (runNow.get()) {
                return -1L;
            }
            return this.internalPool.setTimer(delayMillis, (Handler<Long>)((Handler)timerId -> this.executeAndScheduleNext(key, execCount)));
        });
        if (runNow.get()) {
            this.executeAndScheduleNext(key, execCount);
        }
    }

    private void executeAndScheduleNext(PeriodicTaskKey key, long execCount) {
        Promise runPromise = Promise.promise();
        if (this.activeRuns.computeIfAbsent(key, k -> runPromise.future()) != runPromise) {
            LOGGER.debug("already active. task='{}' execCount={}", (Object)key, (Object)execCount);
            return;
        }
        long startTime = System.nanoTime();
        this.internalPool.executeBlocking(promise -> this.executeInternal((Promise<ScheduleDecision>)promise, key, execCount), false).onComplete(outcome -> {
            DurationSpec delay;
            long priorExecutionDurationMillis;
            LOGGER.debug("Task run finishes. task='{}' outcome={} execCount={}", new Object[]{key, outcome, execCount});
            runPromise.complete();
            this.activeRuns.remove(key);
            if (outcome.result() == ScheduleDecision.RESCHEDULE) {
                priorExecutionDurationMillis = 0L;
                delay = key.task.initialDelay();
            } else {
                priorExecutionDurationMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTime);
                delay = key.task.delay();
            }
            this.schedule(key, priorExecutionDurationMillis, delay.to(TimeUnit.MILLISECONDS), execCount + 1L);
        });
    }

    public Future<Void> unschedule(PeriodicTask task) {
        PeriodicTaskKey key = new PeriodicTaskKey(task);
        AtomicBoolean alreadyUnscheduled = new AtomicBoolean(false);
        Long timerId = this.timerIds.computeIfPresent(key, (k, tid) -> {
            alreadyUnscheduled.set(tid == -2L);
            if (tid > 0L) {
                this.internalPool.cancelTimer((long)tid);
            }
            return -2L;
        });
        if (timerId == null) {
            LOGGER.debug("No such task to unschedule. task='{}'", (Object)key);
            return Future.failedFuture((String)"No such task to unschedule");
        }
        if (alreadyUnscheduled.get()) {
            LOGGER.debug("Task is already unscheduled. task='{}'", (Object)key);
            return Future.failedFuture((String)"Task is already unscheduled");
        }
        LOGGER.debug("Unscheduling task. task='{}' timerId={}", (Object)key, (Object)timerId);
        boolean removeEntry = !this.activeRuns.containsKey(key);
        return this.activeRuns.getOrDefault(key, (Future<Void>)Future.succeededFuture()).andThen(ignored -> {
            try {
                task.close();
            }
            catch (Throwable cause) {
                LOGGER.warn("Failed to close task during unscheduling. task='{}'", (Object)key, (Object)cause);
            }
            if (removeEntry) {
                this.timerIds.remove(key);
            }
        });
    }

    public void close(Promise<Void> completion) {
        LOGGER.info("Closing...");
        try {
            this.timerIds.keySet().forEach(key -> this.unschedule(key.task));
            this.timerIds.clear();
            completion.complete();
        }
        catch (Throwable throwable) {
            completion.fail(throwable);
        }
    }

    private void executeInternal(Promise<ScheduleDecision> promise, PeriodicTaskKey key, long execCount) {
        PeriodicTask periodicTask = key.task;
        ScheduleDecision scheduleDecision = this.consolidateScheduleDecision(periodicTask);
        LOGGER.debug("{} task. task='{}' execCount={}", new Object[]{scheduleDecision, key, execCount});
        if (scheduleDecision == ScheduleDecision.EXECUTE) {
            Promise taskRunPromise = Promise.promise();
            taskRunPromise.future().onComplete(ignored -> promise.tryComplete((Object)ScheduleDecision.EXECUTE));
            try {
                periodicTask.execute(taskRunPromise);
            }
            catch (Throwable throwable) {
                LOGGER.warn("Periodic task failed to execute. task='{}' execCount={}", new Object[]{periodicTask.name(), execCount, throwable});
                taskRunPromise.tryFail(throwable);
            }
        } else {
            promise.tryComplete((Object)scheduleDecision);
        }
    }

    protected ScheduleDecision consolidateScheduleDecision(PeriodicTask periodicTask) {
        ScheduleDecision decisionFromTask = periodicTask.scheduleDecision();
        if (decisionFromTask != ScheduleDecision.SKIP && periodicTask instanceof ExecuteOnClusterLeaseholderOnly) {
            return this.clusterLease.toScheduleDecision();
        }
        return decisionFromTask;
    }

    @VisibleForTesting
    Map<PeriodicTaskKey, Long> timerIds() {
        return this.timerIds;
    }

    @VisibleForTesting
    Map<PeriodicTaskKey, Future<Void>> activeRuns() {
        return this.activeRuns;
    }

    static class PeriodicTaskKey {
        private final String fqcnAndName;
        private final PeriodicTask task;

        PeriodicTaskKey(PeriodicTask task) {
            this.fqcnAndName = task.getClass().getCanonicalName() + task.name();
            this.task = task;
        }

        public int hashCode() {
            return this.fqcnAndName.hashCode();
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj instanceof PeriodicTaskKey) {
                return ((PeriodicTaskKey)obj).fqcnAndName.equals(this.fqcnAndName);
            }
            return false;
        }

        public String toString() {
            return this.fqcnAndName;
        }
    }
}

