/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.sidecar.restore;

import com.codahale.metrics.Histogram;
import com.codahale.metrics.Timer;
import com.google.common.annotations.VisibleForTesting;
import io.vertx.core.Future;
import io.vertx.core.Promise;
import io.vertx.ext.web.handler.HttpException;
import java.io.File;
import java.io.IOException;
import java.math.BigInteger;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import org.apache.cassandra.sidecar.cluster.locator.LocalTokenRangesProvider;
import org.apache.cassandra.sidecar.common.data.RestoreJobStatus;
import org.apache.cassandra.sidecar.common.data.SSTableImportOptions;
import org.apache.cassandra.sidecar.common.server.cluster.locator.TokenRange;
import org.apache.cassandra.sidecar.common.server.utils.ThrowableUtils;
import org.apache.cassandra.sidecar.common.utils.Preconditions;
import org.apache.cassandra.sidecar.concurrent.TaskExecutorPool;
import org.apache.cassandra.sidecar.db.RestoreJob;
import org.apache.cassandra.sidecar.db.RestoreRange;
import org.apache.cassandra.sidecar.db.RestoreRangeDatabaseAccessor;
import org.apache.cassandra.sidecar.exceptions.RestoreJobException;
import org.apache.cassandra.sidecar.exceptions.RestoreJobExceptions;
import org.apache.cassandra.sidecar.exceptions.RestoreJobFatalException;
import org.apache.cassandra.sidecar.metrics.DeltaGauge;
import org.apache.cassandra.sidecar.metrics.RestoreMetrics;
import org.apache.cassandra.sidecar.metrics.SidecarMetrics;
import org.apache.cassandra.sidecar.metrics.StopWatch;
import org.apache.cassandra.sidecar.metrics.instance.InstanceMetrics;
import org.apache.cassandra.sidecar.restore.RestoreJobUtil;
import org.apache.cassandra.sidecar.restore.RestoreRangeHandler;
import org.apache.cassandra.sidecar.restore.RestoreSliceManifest;
import org.apache.cassandra.sidecar.restore.StorageClient;
import org.apache.cassandra.sidecar.utils.AsyncFileSystemUtils;
import org.apache.cassandra.sidecar.utils.SSTableImporter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.core.exception.ApiCallTimeoutException;
import software.amazon.awssdk.services.s3.model.NoSuchKeyException;
import software.amazon.awssdk.services.s3.model.S3Exception;

public class RestoreRangeTask
implements RestoreRangeHandler {
    private static final Logger LOGGER = LoggerFactory.getLogger(RestoreRangeTask.class);
    private final RestoreRange range;
    private final StorageClient s3Client;
    private final TaskExecutorPool executorPool;
    private final SSTableImporter importer;
    private final double requiredUsableSpacePercentage;
    private final RestoreRangeDatabaseAccessor rangeDatabaseAccessor;
    private final RestoreJobUtil restoreJobUtil;
    private final LocalTokenRangesProvider localTokenRangesProvider;
    private final RestoreMetrics metrics;
    private final InstanceMetrics instanceMetrics;
    private long taskStartTimeNanos = -1L;

    public RestoreRangeTask(RestoreRange range, StorageClient s3Client, TaskExecutorPool executorPool, SSTableImporter importer, double requiredUsableSpacePercentage, RestoreRangeDatabaseAccessor rangeDatabaseAccessor, RestoreJobUtil restoreJobUtil, LocalTokenRangesProvider localTokenRangesProvider, SidecarMetrics metrics) {
        Preconditions.checkArgument((!range.job().isManagedBySidecar() || rangeDatabaseAccessor != null ? 1 : 0) != 0, (String)"rangeDatabaseAccessor cannot be null");
        this.range = range;
        this.s3Client = s3Client;
        this.executorPool = executorPool;
        this.importer = importer;
        this.requiredUsableSpacePercentage = requiredUsableSpacePercentage;
        this.rangeDatabaseAccessor = rangeDatabaseAccessor;
        this.restoreJobUtil = restoreJobUtil;
        this.localTokenRangesProvider = localTokenRangesProvider;
        this.metrics = metrics.server().restore();
        this.instanceMetrics = metrics.instance(range.owner().id());
    }

    public static RestoreRangeHandler failed(RestoreJobException cause, RestoreRange range) {
        return new Failed(cause, range);
    }

    @Override
    public long elapsedInNanos() {
        return this.taskStartTimeNanos == -1L ? -1L : this.currentTimeInNanos() - this.taskStartTimeNanos;
    }

    @Override
    public RestoreRange range() {
        return this.range;
    }

    public void handle(Promise<RestoreRange> event) {
        this.taskStartTimeNanos = this.restoreJobUtil.currentTimeNanos();
        RestoreRangeTask.failOnCancelled(this.range, null).compose(v -> AsyncFileSystemUtils.ensureSufficientStorage(this.range.stageDirectory().toString(), this.range.estimatedSpaceRequiredInBytes(), this.requiredUsableSpacePercentage, this.executorPool)).compose(v -> RestoreRangeTask.failOnCancelled(this.range, v)).compose(v -> {
            RestoreJob job = this.range.job();
            if (job.isManagedBySidecar()) {
                if (job.status == RestoreJobStatus.STAGE_READY) {
                    if (Files.exists(this.range.stagedObjectPath(), new LinkOption[0])) {
                        LOGGER.debug("The slice has been staged already. sliceKey={} stagedFilePath={}", (Object)this.range.sliceKey(), (Object)this.range.stagedObjectPath());
                        this.range.completeStagePhase();
                        this.rangeDatabaseAccessor.updateStatus(this.range);
                        return Future.succeededFuture();
                    }
                    return RestoreRangeTask.failOnCancelled(this.range, null).compose(value -> this.checkObjectExistence()).compose(value -> RestoreRangeTask.failOnCancelled(this.range, value)).compose(headObject -> this.downloadSlice()).compose(value -> RestoreRangeTask.failOnCancelled(this.range, value)).compose(file -> {
                        this.range.completeStagePhase();
                        this.rangeDatabaseAccessor.updateStatus(this.range);
                        return Future.succeededFuture();
                    });
                }
                if (job.status == RestoreJobStatus.IMPORT_READY) {
                    return this.unzipAndImport(this.range.stagedObjectPath().toFile(), () -> this.rangeDatabaseAccessor.updateStatus(this.range));
                }
                String msg = "Unexpected restore job status. Expected only STAGE_READY or IMPORT_READY when processing active slices. Found status: " + job.statusWithOptionalDescription();
                IllegalStateException unexpectedState = new IllegalStateException(msg);
                return Future.failedFuture((Throwable)RestoreJobExceptions.ofFatal("Unexpected restore job status", this.range, unexpectedState));
            }
            return this.downloadSliceAndImport();
        }).onSuccess(v -> event.tryComplete((Object)this.range)).onFailure(cause -> event.tryFail((Throwable)RestoreJobExceptions.propagate(cause)));
    }

    private Future<Void> downloadSliceAndImport() {
        return RestoreRangeTask.failOnCancelled(this.range, null).compose(value -> this.checkObjectExistence()).compose(value -> RestoreRangeTask.failOnCancelled(this.range, value)).compose(value -> this.downloadSlice()).compose(value -> RestoreRangeTask.failOnCancelled(this.range, value)).compose(this::unzipAndImport);
    }

    private Future<?> checkObjectExistence() {
        if (this.range.existsOnS3()) {
            LOGGER.debug("The slice already exists on S3. jobId={} sliceKey={}", (Object)this.range.jobId(), (Object)this.range.sliceKey());
            return Future.succeededFuture();
        }
        return Future.fromCompletionStage(this.s3Client.objectExists(this.range)).compose(headObjectResponse -> {
            long durationNanos = this.currentTimeInNanos() - this.range.sliceCreationTimeNanos();
            ((Timer)this.metrics.sliceReplicationTime.metric).update(durationNanos, TimeUnit.NANOSECONDS);
            this.range.setExistsOnS3(headObjectResponse.contentLength());
            LOGGER.debug("Slice is now available on S3. jobId={} sliceKey={} replicationTimeNanos={}", new Object[]{this.range.jobId(), this.range.sliceKey(), durationNanos});
            return Future.succeededFuture();
        }, cause -> Future.failedFuture((Throwable)this.toRestoreJobException((Throwable)cause)));
    }

    private RestoreJobException toRestoreJobException(Throwable cause) {
        S3Exception s3Exception = (S3Exception)ThrowableUtils.getCause((Throwable)cause, S3Exception.class);
        if (s3Exception == null) {
            return RestoreJobExceptions.ofFatal("Unexpected error when checking object existence", this.range, cause);
        }
        if (s3Exception instanceof NoSuchKeyException) {
            return RestoreJobExceptions.of("Object not found", this.range, s3Exception.awsErrorDetails(), null);
        }
        if (s3Exception.statusCode() == 400 && s3Exception.awsErrorDetails().errorCode().equalsIgnoreCase("ExpiredToken")) {
            ((DeltaGauge)this.metrics.tokenExpired.metric).update(1L);
            return RestoreJobExceptions.ofFatal("Token has expired", this.range, s3Exception.awsErrorDetails(), (Throwable)s3Exception);
        }
        if (s3Exception.statusCode() == 403) {
            ((DeltaGauge)this.metrics.tokenUnauthorized.metric).update(1L);
            return RestoreJobExceptions.ofFatal("Object access is forbidden", this.range, s3Exception.awsErrorDetails(), (Throwable)s3Exception);
        }
        if (s3Exception.statusCode() == 404) {
            if (s3Exception.awsErrorDetails().errorCode().equalsIgnoreCase("NoSuchKey")) {
                return RestoreJobExceptions.of("Object not found", this.range, s3Exception.awsErrorDetails(), null);
            }
            if (s3Exception.awsErrorDetails().errorCode().equalsIgnoreCase("NoSuchBucket")) {
                return RestoreJobExceptions.ofFatal("Bucket not found", this.range, s3Exception.awsErrorDetails(), null);
            }
        } else if (s3Exception.statusCode() == 412) {
            ((DeltaGauge)this.instanceMetrics.restore().sliceChecksumMismatches.metric).update(1L);
            return RestoreJobExceptions.ofFatal("Object checksum mismatched", this.range, s3Exception.awsErrorDetails(), (Throwable)s3Exception);
        }
        return RestoreJobExceptions.of("Unable to check object existence. ", this.range, s3Exception.awsErrorDetails(), (Throwable)s3Exception);
    }

    private long currentTimeInNanos() {
        return this.restoreJobUtil.currentTimeNanos();
    }

    private Future<File> downloadSlice() {
        if (this.range.downloadAttempt() > 0) {
            LOGGER.debug("Retrying downloading slice. sliceKey={}", (Object)this.range.sliceKey());
            ((DeltaGauge)this.instanceMetrics.restore().sliceDownloadRetries.metric).update(1L);
        }
        LOGGER.info("Begin downloading restore slice. sliceKey={}", (Object)this.range.sliceKey());
        Future future = this.s3Client.downloadObjectIfAbsent(this.range, this.executorPool).recover(cause -> {
            LOGGER.warn("Failed to download restore slice. sliceKey={}", (Object)this.range.sliceKey(), cause);
            this.range.incrementDownloadAttempt();
            if (ThrowableUtils.getCause((Throwable)cause, ApiCallTimeoutException.class) != null) {
                LOGGER.warn("Downloading restore slice times out. sliceKey={}", (Object)this.range.sliceKey());
                ((DeltaGauge)this.instanceMetrics.restore().sliceDownloadTimeouts.metric).update(1L);
                return Future.failedFuture((Throwable)RestoreJobExceptions.of("Download object times out. Retry later", this.range, cause));
            }
            return Future.failedFuture((Throwable)RestoreJobExceptions.ofFatal("Unrecoverable error when downloading object", this.range, cause));
        });
        return StopWatch.measureTimeTaken(future, duration -> {
            LOGGER.info("Finish downloading restore slice. sliceKey={}", (Object)this.range.sliceKey());
            ((Timer)this.instanceMetrics.restore().sliceDownloadTime.metric).update(duration, TimeUnit.NANOSECONDS);
            ((Histogram)this.instanceMetrics.restore().sliceCompressedSizeInBytes.metric).update(this.range.sliceCompressedSize());
            ((Histogram)this.instanceMetrics.restore().sliceUncompressedSizeInBytes.metric).update(this.range.sliceUncompressedSize());
        });
    }

    @VisibleForTesting
    Future<Void> unzipAndImport(File file) {
        return this.unzipAndImport(file, null);
    }

    Future<Void> unzipAndImport(File file, Runnable onSuccessCommit) {
        if (file == null) {
            return Future.failedFuture((Throwable)RestoreJobExceptions.ofFatal("Slice not found from disk", this.range, null));
        }
        return RestoreRangeTask.failOnCancelled(this.range, file).compose(this::unzip).compose(value -> RestoreRangeTask.failOnCancelled(this.range, value)).compose(this::validateFiles).compose(value -> RestoreRangeTask.failOnCancelled(this.range, value)).compose(this::commit).compose(success -> {
            this.range.completeImportPhase();
            if (onSuccessCommit == null) {
                return Future.succeededFuture();
            }
            return this.executorPool.runBlocking(onSuccessCommit::run);
        }, failure -> {
            this.logWarnIfHasHttpExceptionCauseOnCommit((Throwable)failure, this.range);
            return Future.failedFuture((Throwable)RestoreJobExceptions.propagate("Fail to commit range. " + this.range.shortDescription(), failure));
        });
    }

    private Future<File> unzip(File zipFile) {
        Future<File> future = this.executorPool.executeBlocking(() -> this.unzipAction(zipFile), false);
        return StopWatch.measureTimeTaken(future, d -> ((Timer)this.instanceMetrics.restore().sliceUnzipTime.metric).update(d, TimeUnit.NANOSECONDS));
    }

    File unzipAction(File zipFile) throws RestoreJobException {
        File targetDir = this.range.stageDirectory().resolve(this.range.keyspace()).resolve(this.range.table()).toFile();
        boolean targetDirExist = targetDir.isDirectory();
        if (!zipFile.exists()) {
            if (targetDirExist) {
                LOGGER.debug("The files in slice are already extracted. Maybe it is a retried task? jobId={} sliceKey={}", (Object)this.range.jobId(), (Object)this.range.sliceKey());
                return targetDir;
            }
            throw new RestoreJobException("Object not found from disk. File: " + zipFile);
        }
        try {
            Files.createDirectories(targetDir.toPath(), new FileAttribute[0]);
            RestoreJobUtil.cleanDirectory(targetDir.toPath());
            RestoreJobUtil.unzip(zipFile, targetDir);
            if (!zipFile.delete()) {
                LOGGER.warn("File deletion attempt failed. jobId={} sliceKey={} file={}", new Object[]{this.range.jobId(), this.range.sliceKey(), zipFile.getAbsolutePath()});
            }
            return targetDir;
        }
        catch (Exception cause) {
            throw RestoreJobExceptions.propagate("Failed to unzip. File: " + zipFile, cause);
        }
    }

    private Future<File> validateFiles(File directory) {
        Future<File> future = this.executorPool.executeBlocking(() -> this.validateFilesAction(directory), false);
        return StopWatch.measureTimeTaken(future, d -> ((Timer)this.instanceMetrics.restore().sliceValidationTime.metric).update(d, TimeUnit.NANOSECONDS));
    }

    File validateFilesAction(File directory) throws RestoreJobException, IOException {
        File manifestFile = new File(directory, "manifest.json");
        RestoreSliceManifest manifest = RestoreSliceManifest.read(manifestFile);
        if (manifest.isEmpty()) {
            throw new RestoreJobFatalException("The downloaded slice has no data. Directory: " + directory);
        }
        if (this.range.job().isManagedBySidecar()) {
            this.removeOutOfRangeSSTables(directory, manifest);
        }
        Map<String, String> checksums = manifest.mergeAllChecksums();
        File[] files = directory.listFiles((dir, name) -> !name.equals("manifest.json"));
        if (files == null || files.length != checksums.size()) {
            String msg = "Number of files does not match. Expected: " + checksums.size() + "; Actual: " + (files == null ? 0 : files.length) + "; Directory: " + directory;
            throw new RestoreJobFatalException(msg);
        }
        this.compareChecksums(checksums, files);
        for (File file : files) {
            if (!file.getName().endsWith("-Data.db")) continue;
            ((Histogram)this.instanceMetrics.restore().dataSSTableComponentSize.metric).update(file.length());
        }
        return directory;
    }

    private void removeOutOfRangeSSTables(File directory, RestoreSliceManifest manifest) throws RestoreJobException, IOException {
        Set<TokenRange> ranges = this.localTokenRangesProvider.localTokenRanges(this.range.keyspace()).get(this.range.owner().id());
        if (ranges == null || ranges.isEmpty()) {
            throw new RestoreJobException("Unable to fetch local range, retry later");
        }
        Iterator it = manifest.entrySet().iterator();
        while (it.hasNext()) {
            RestoreSliceManifest.ManifestEntry entry = (RestoreSliceManifest.ManifestEntry)it.next().getValue();
            TokenRange sstableRange = new TokenRange(entry.startToken().subtract(BigInteger.ONE), entry.endToken());
            boolean hasOverlap = false;
            boolean fullyEnclosed = false;
            for (TokenRange owningRange : ranges) {
                if (hasOverlap) break;
                hasOverlap = owningRange.intersects(sstableRange);
                if (!hasOverlap) continue;
                fullyEnclosed = owningRange.encloses(sstableRange);
            }
            if (!hasOverlap) {
                it.remove();
                for (String fileName : entry.componentsChecksum().keySet()) {
                    Path path = directory.toPath().resolve(fileName);
                    Files.deleteIfExists(path);
                }
                continue;
            }
            if (fullyEnclosed) continue;
            this.range.requestOutOfRangeDataCleanup();
        }
    }

    private void compareChecksums(Map<String, String> expectedChecksums, File[] files) throws RestoreJobFatalException {
        for (File file : files) {
            String name = file.getName();
            String expectedChecksum = expectedChecksums.get(name);
            if (expectedChecksum == null) {
                throw new RestoreJobFatalException("File not found in manifest. File: " + name);
            }
            try {
                String actualChecksum = this.restoreJobUtil.checksum(file);
                if (actualChecksum.equals(expectedChecksum)) continue;
                String msg = "Checksum does not match. Expected: " + expectedChecksum + "; actual: " + actualChecksum + "; file: " + file;
                throw new RestoreJobFatalException(msg);
            }
            catch (IOException cause) {
                throw new RestoreJobFatalException("Failed to calculate checksum. File: " + file, cause);
            }
        }
    }

    Future<Void> commit(File directory) {
        LOGGER.info("Begin committing SSTables. jobId={} sliceKey={}", (Object)this.range.jobId(), (Object)this.range.sliceKey());
        SSTableImportOptions options = this.range.job().importOptions;
        SSTableImporter.ImportOptions importOptions = new SSTableImporter.ImportOptions.Builder().host(this.range.owner().host()).keyspace(this.range.keyspace()).tableName(this.range.table()).directory(directory.toString()).resetLevel(options.resetLevel()).clearRepaired(options.clearRepaired()).verifySSTables(options.verifySSTables()).verifyTokens(options.verifyTokens()).invalidateCaches(options.invalidateCaches()).extendedVerify(options.extendedVerify()).copyData(options.copyData()).uploadId(this.range.uploadId()).build();
        Future future = this.importer.scheduleImport(importOptions).onSuccess(ignored -> LOGGER.info("Finish committing SSTables. jobId={} sliceKey={}", (Object)this.range.jobId(), (Object)this.range.sliceKey()));
        return StopWatch.measureTimeTaken(future, d -> ((Timer)this.instanceMetrics.restore().sliceImportTime.metric).update(d, TimeUnit.NANOSECONDS));
    }

    static <T> Future<T> failOnCancelled(RestoreRange range, T value) {
        if (range.isCancelled()) {
            return Future.failedFuture((Throwable)RestoreJobExceptions.ofFatal("Restore range is cancelled", range, null));
        }
        return Future.succeededFuture(value);
    }

    private void logWarnIfHasHttpExceptionCauseOnCommit(Throwable throwable, RestoreRange range) {
        HttpException httpException = (HttpException)ThrowableUtils.getCause((Throwable)throwable, HttpException.class);
        if (httpException == null) {
            return;
        }
        LOGGER.warn("Committing range failed with HttpException. jobId={} startToken={} endToken={} sliceKey={} statusCode={} exceptionPayload={}", new Object[]{range.jobId(), range.startToken(), range.endToken(), range.sliceKey(), httpException.getStatusCode(), httpException.getPayload(), httpException});
    }

    @VisibleForTesting
    void removeOutOfRangeSSTablesUnsafe(File directory, RestoreSliceManifest manifest) throws RestoreJobException, IOException {
        this.removeOutOfRangeSSTables(directory, manifest);
    }

    @VisibleForTesting
    void compareChecksumsUnsafe(Map<String, String> expectedChecksums, File[] files) throws RestoreJobFatalException {
        this.compareChecksums(expectedChecksums, files);
    }

    public static class Failed
    implements RestoreRangeHandler {
        private final RestoreJobException cause;
        private final RestoreRange range;

        public Failed(RestoreJobException cause, RestoreRange range) {
            this.cause = cause;
            this.range = range;
        }

        public void handle(Promise<RestoreRange> promise) {
            promise.tryFail((Throwable)this.cause);
        }

        @Override
        public long elapsedInNanos() {
            return 0L;
        }

        @Override
        public RestoreRange range() {
            return this.range;
        }
    }
}

