/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.sidecar.restore;

import com.codahale.metrics.DefaultSettableGauge;
import com.datastax.driver.core.LocalDate;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Sets;
import com.google.inject.Inject;
import com.google.inject.Provider;
import com.google.inject.Singleton;
import io.vertx.core.Promise;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import org.apache.cassandra.sidecar.cluster.instance.InstanceMetadata;
import org.apache.cassandra.sidecar.cluster.locator.LocalTokenRangesProvider;
import org.apache.cassandra.sidecar.common.data.RestoreJobStatus;
import org.apache.cassandra.sidecar.common.response.TokenRangeReplicasResponse;
import org.apache.cassandra.sidecar.common.server.cluster.locator.TokenRange;
import org.apache.cassandra.sidecar.common.server.utils.DurationSpec;
import org.apache.cassandra.sidecar.common.utils.Preconditions;
import org.apache.cassandra.sidecar.concurrent.ExecutorPools;
import org.apache.cassandra.sidecar.concurrent.TaskExecutorPool;
import org.apache.cassandra.sidecar.config.RestoreJobConfiguration;
import org.apache.cassandra.sidecar.config.SidecarConfiguration;
import org.apache.cassandra.sidecar.db.RestoreJob;
import org.apache.cassandra.sidecar.db.RestoreJobDatabaseAccessor;
import org.apache.cassandra.sidecar.db.RestoreRange;
import org.apache.cassandra.sidecar.db.RestoreRangeDatabaseAccessor;
import org.apache.cassandra.sidecar.db.RestoreSlice;
import org.apache.cassandra.sidecar.db.RestoreSliceDatabaseAccessor;
import org.apache.cassandra.sidecar.db.schema.SidecarSchema;
import org.apache.cassandra.sidecar.exceptions.RestoreJobFatalException;
import org.apache.cassandra.sidecar.metrics.RestoreMetrics;
import org.apache.cassandra.sidecar.metrics.SidecarMetrics;
import org.apache.cassandra.sidecar.restore.RestoreJobManagerGroup;
import org.apache.cassandra.sidecar.restore.RestoreJobProgressTracker;
import org.apache.cassandra.sidecar.restore.RestoreJobUtil;
import org.apache.cassandra.sidecar.restore.RingTopologyChangeListener;
import org.apache.cassandra.sidecar.restore.RingTopologyRefresher;
import org.apache.cassandra.sidecar.tasks.PeriodicTask;
import org.apache.cassandra.sidecar.tasks.PeriodicTaskExecutor;
import org.apache.cassandra.sidecar.tasks.ScheduleDecision;
import org.apache.cassandra.sidecar.utils.InstanceMetadataFetcher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class RestoreJobDiscoverer
implements PeriodicTask,
RingTopologyChangeListener {
    private static final Logger LOGGER = LoggerFactory.getLogger(RestoreJobDiscoverer.class);
    private final RestoreJobConfiguration restoreJobConfig;
    private final SidecarSchema sidecarSchema;
    private final RestoreJobDatabaseAccessor restoreJobDatabaseAccessor;
    private final RestoreSliceDatabaseAccessor restoreSliceDatabaseAccessor;
    private final RestoreRangeDatabaseAccessor restoreRangeDatabaseAccessor;
    private final Provider<RestoreJobManagerGroup> restoreJobManagerGroupSingleton;
    private final LocalTokenRangesProvider localTokenRangesProvider;
    private final InstanceMetadataFetcher instanceMetadataFetcher;
    private final RestoreMetrics metrics;
    private final JobIdsByDay jobIdsByDay;
    private final RingTopologyRefresher ringTopologyRefresher;
    private final AtomicBoolean isExecuting = new AtomicBoolean(false);
    private final TaskExecutorPool executorPool;
    private int inflightJobsCount = 0;
    private int jobDiscoveryRecencyDays;
    private PeriodicTaskExecutor periodicTaskExecutor;

    @Inject
    public RestoreJobDiscoverer(SidecarConfiguration config, SidecarSchema sidecarSchema, RestoreJobDatabaseAccessor restoreJobDatabaseAccessor, RestoreSliceDatabaseAccessor restoreSliceDatabaseAccessor, RestoreRangeDatabaseAccessor restoreRangeDatabaseAccessor, Provider<RestoreJobManagerGroup> restoreJobManagerGroupProvider, InstanceMetadataFetcher instanceMetadataFetcher, RingTopologyRefresher ringTopologyRefresher, ExecutorPools executorPools, SidecarMetrics metrics) {
        this(config.restoreJobConfiguration(), sidecarSchema, restoreJobDatabaseAccessor, restoreSliceDatabaseAccessor, restoreRangeDatabaseAccessor, restoreJobManagerGroupProvider, instanceMetadataFetcher, ringTopologyRefresher, executorPools, metrics);
    }

    @VisibleForTesting
    RestoreJobDiscoverer(RestoreJobConfiguration restoreJobConfig, SidecarSchema sidecarSchema, RestoreJobDatabaseAccessor restoreJobDatabaseAccessor, RestoreSliceDatabaseAccessor restoreSliceDatabaseAccessor, RestoreRangeDatabaseAccessor restoreRangeDatabaseAccessor, Provider<RestoreJobManagerGroup> restoreJobManagerGroupProvider, InstanceMetadataFetcher instanceMetadataFetcher, RingTopologyRefresher ringTopologyRefresher, ExecutorPools executorPools, SidecarMetrics metrics) {
        this.restoreJobConfig = restoreJobConfig;
        this.sidecarSchema = sidecarSchema;
        this.restoreJobDatabaseAccessor = restoreJobDatabaseAccessor;
        this.restoreSliceDatabaseAccessor = restoreSliceDatabaseAccessor;
        this.restoreRangeDatabaseAccessor = restoreRangeDatabaseAccessor;
        this.jobDiscoveryRecencyDays = restoreJobConfig.jobDiscoveryMinimumRecencyDays();
        this.restoreJobManagerGroupSingleton = restoreJobManagerGroupProvider;
        this.instanceMetadataFetcher = instanceMetadataFetcher;
        this.ringTopologyRefresher = ringTopologyRefresher;
        this.localTokenRangesProvider = ringTopologyRefresher;
        this.metrics = metrics.server().restore();
        this.jobIdsByDay = new JobIdsByDay();
        this.executorPool = executorPools.internal();
    }

    @Override
    public ScheduleDecision scheduleDecision() {
        return this.shouldSkip() ? ScheduleDecision.SKIP : ScheduleDecision.EXECUTE;
    }

    @Override
    public DurationSpec delay() {
        return this.hasInflightJobs() ? this.restoreJobConfig.jobDiscoveryActiveLoopDelay() : this.restoreJobConfig.jobDiscoveryIdleLoopDelay();
    }

    @Override
    public void execute(Promise<Void> promise) {
        this.tryExecuteDiscovery();
        promise.tryComplete();
    }

    public void tryExecuteDiscovery() {
        if (!this.isExecuting.compareAndSet(false, true)) {
            LOGGER.debug("Another thread is executing the restore job discovery already. Skipping...");
            return;
        }
        try {
            this.executeInternal();
        }
        finally {
            this.isExecuting.set(false);
        }
    }

    private boolean shouldSkip() {
        boolean shouldSkip;
        boolean bl = shouldSkip = !this.sidecarSchema.isInitialized();
        if (shouldSkip) {
            LOGGER.trace("Skipping restore job discovering due to sidecarSchema not initialized");
        }
        boolean executing = this.isExecuting.get();
        boolean bl2 = shouldSkip = shouldSkip || executing;
        if (executing) {
            LOGGER.trace("Skipping restore job discovering due to overlapping execution of this task");
        }
        return shouldSkip;
    }

    private void executeInternal() {
        Preconditions.checkState((this.periodicTaskExecutor != null ? 1 : 0) != 0, (String)"Loop executor is not registered");
        LOGGER.info("Discovering restore jobs. inflightJobsCount={} delayMs={} jobDiscoveryRecencyDays={}", new Object[]{this.inflightJobsCount, this.delay(), this.jobDiscoveryRecencyDays});
        this.inflightJobsCount = 0;
        RunContext context = new RunContext();
        List<RestoreJob> restoreJobs = this.restoreJobDatabaseAccessor.findAllRecent(context.nowMillis, this.jobDiscoveryRecencyDays);
        RestoreJobManagerGroup restoreJobManagers = (RestoreJobManagerGroup)this.restoreJobManagerGroupSingleton.get();
        for (RestoreJob job : restoreJobs) {
            try {
                this.processOneJob(job, restoreJobManagers, context);
            }
            catch (Exception exception) {
                LOGGER.warn("Exception on processing job. jobId: {}", (Object)job.jobId, (Object)exception);
            }
        }
        this.jobIdsByDay.cleanupMaybe();
        this.jobDiscoveryRecencyDays = Math.max(context.earliestInDays, this.restoreJobConfig.jobDiscoveryMinimumRecencyDays());
        LOGGER.info("Exit job discovery. inflightJobsCount={} jobDiscoveryRecencyDays={} expiredJobs={} abortedJobs={}", new Object[]{this.inflightJobsCount, this.jobDiscoveryRecencyDays, context.expiredJobs, context.abortedJobs});
        ((DefaultSettableGauge)this.metrics.activeJobs.metric).setValue((Object)this.inflightJobsCount);
    }

    @Override
    public void registerPeriodicTaskExecutor(PeriodicTaskExecutor executor) {
        this.periodicTaskExecutor = executor;
    }

    @Override
    public void onRingTopologyChanged(String keyspace, TokenRangeReplicasResponse oldTopology, TokenRangeReplicasResponse newTopology) {
        Map<Integer, Set<TokenRange>> localRangesFromNew;
        if (oldTopology == null) {
            LOGGER.debug("Received RingTopologyChanged notification for new topology discovered. It is already handled inline at findSlicesAndSubmit. Exiting early. keyspace={}", (Object)keyspace);
            return;
        }
        Map<Integer, Set<TokenRange>> localRangesFromOld = RingTopologyRefresher.calculateLocalTokenRanges(this.instanceMetadataFetcher, oldTopology);
        if (Objects.equals(localRangesFromOld, localRangesFromNew = RingTopologyRefresher.calculateLocalTokenRanges(this.instanceMetadataFetcher, newTopology))) {
            LOGGER.debug("Local token ranges derived from both topology are the same. No need to update restore ranges.");
            return;
        }
        HashMap<Integer, Set> lostRanges = new HashMap<Integer, Set>(localRangesFromNew.size());
        HashMap<Integer, Set> gainedRanges = new HashMap<Integer, Set>(localRangesFromNew.size());
        for (Integer instanceId2 : Sets.union(localRangesFromOld.keySet(), localRangesFromNew.keySet())) {
            Set<TokenRange> rangesFromOld = localRangesFromOld.get(instanceId2);
            Set<TokenRange> rangesFromNew = localRangesFromNew.get(instanceId2);
            Preconditions.checkState((rangesFromNew != null || rangesFromOld != null ? 1 : 0) != 0, (String)("Token ranges of instance: " + instanceId2 + " do not exist in both old and new"));
            if (rangesFromOld == null) {
                gainedRanges.put(instanceId2, rangesFromNew);
                continue;
            }
            if (rangesFromNew == null) {
                lostRanges.put(instanceId2, rangesFromOld);
                continue;
            }
            TokenRange.SymmetricDiffResult symmetricDiffResult = TokenRange.symmetricDiff(rangesFromOld, rangesFromNew);
            lostRanges.put(instanceId2, symmetricDiffResult.onlyInLeft);
            gainedRanges.put(instanceId2, symmetricDiffResult.onlyInRight);
        }
        Set<UUID> jobIds = this.ringTopologyRefresher.allRestoreJobsOfKeyspace(keyspace);
        for (UUID jobId : jobIds) {
            RestoreJob restoreJob = this.restoreJobDatabaseAccessor.find(jobId);
            if (restoreJob == null) continue;
            try {
                lostRanges.forEach((instanceId, ranges) -> this.discardLostRanges(restoreJob, (int)instanceId, (Set<TokenRange>)ranges));
                gainedRanges.forEach((instanceId, ranges) -> this.findSlicesOfCassandraNodeAndSubmit(restoreJob, (int)instanceId, (Set<TokenRange>)ranges));
            }
            catch (Exception e) {
                LOGGER.warn("Unexpected exception when adjusting restore job ranges. jobId={}", (Object)jobId, (Object)e);
            }
        }
    }

    private void processOneJob(RestoreJob job, RestoreJobManagerGroup restoreJobManagers, RunContext context) {
        if (this.jobIdsByDay.shouldLogJob(job)) {
            LOGGER.info("Found job. jobId={} job={}", (Object)job.jobId, (Object)job);
        }
        switch (job.status) {
            case STAGED: {
                this.jobIdsByDay.unsetSlicesDiscovered(job);
            }
            case CREATED: 
            case STAGE_READY: 
            case IMPORT_READY: {
                if (job.hasExpired(context.nowMillis)) {
                    this.abortExpiredJob(job, restoreJobManagers, context);
                    break;
                }
                context.earliestInDays = Math.max(context.earliestInDays, this.delta(context.today, job.createdAt));
                restoreJobManagers.updateRestoreJob(job);
                this.processSidecarManagedJobMaybe(job);
                ++this.inflightJobsCount;
                break;
            }
            case FAILED: 
            case ABORTED: 
            case SUCCEEDED: {
                this.finalizeJob(restoreJobManagers, job);
                break;
            }
            default: {
                LOGGER.warn("Encountered unknown job status. jobId={} status={}", (Object)job.jobId, (Object)job.status);
            }
        }
    }

    private void abortExpiredJob(RestoreJob job, RestoreJobManagerGroup restoreJobManagers, RunContext context) {
        ++context.expiredJobs;
        boolean aborted = this.abortJob(job);
        if (aborted) {
            ++context.abortedJobs;
            this.finalizeJob(restoreJobManagers, job);
        }
    }

    private void processSidecarManagedJobMaybe(RestoreJob job) {
        if (!job.isManagedBySidecar()) {
            return;
        }
        this.ringTopologyRefresher.register(job, this);
        if (this.shouldFindSlicesAndSubmit(job)) {
            this.findSlicesAndSubmit(job);
            this.jobIdsByDay.markSlicesDiscovered(job);
        }
    }

    private boolean shouldFindSlicesAndSubmit(RestoreJob job) {
        return (job.status == RestoreJobStatus.STAGE_READY || job.status == RestoreJobStatus.IMPORT_READY) && !this.jobIdsByDay.isSliceDiscovered(job);
    }

    private void finalizeJob(RestoreJobManagerGroup restoreJobManagers, RestoreJob job) {
        restoreJobManagers.removeJobInternal(job);
        if (job.isManagedBySidecar()) {
            this.ringTopologyRefresher.unregister(job, this);
        }
    }

    private void findSlicesAndSubmit(RestoreJob restoreJob) {
        this.localTokenRangesProvider.localTokenRanges(restoreJob.keyspaceName, true).forEach((instanceId, ranges) -> this.findSlicesOfCassandraNodeAndSubmit(restoreJob, (int)instanceId, (Set<TokenRange>)ranges));
    }

    private void findSlicesOfCassandraNodeAndSubmit(RestoreJob restoreJob, int instanceId, Set<TokenRange> ranges) {
        InstanceMetadata instance = this.instanceMetadataFetcher.instance(instanceId);
        ranges.forEach(range -> this.findSlicesOfRangeAndSubmit(instance, restoreJob, (TokenRange)range));
    }

    private void findSlicesOfRangeAndSubmit(InstanceMetadata instance, RestoreJob restoreJob, TokenRange range) {
        short bucketId = 0;
        this.restoreSliceDatabaseAccessor.selectByJobByBucketByTokenRange(restoreJob, bucketId, range).forEach(slice -> {
            RestoreSlice trimmed = slice.trimMaybe(range);
            String uploadId = RestoreJobUtil.generateUniqueUploadId(trimmed.jobId(), trimmed.sliceId());
            RestoreRange restoreRange = RestoreRange.builderFromSlice(trimmed).ownerInstance(instance).stageDirectory(Paths.get(instance.stagingDir(), new String[0]), uploadId).build();
            RestoreJobProgressTracker.Status status = this.submit(instance, restoreJob, restoreRange);
            if (status == RestoreJobProgressTracker.Status.CREATED) {
                this.restoreRangeDatabaseAccessor.create(restoreRange);
            }
        });
    }

    private void discardLostRanges(RestoreJob restoreJob, int instanceId, Set<TokenRange> otherRanges) {
        InstanceMetadata instance = this.instanceMetadataFetcher.instance(instanceId);
        RestoreJobManagerGroup managerGroup = (RestoreJobManagerGroup)this.restoreJobManagerGroupSingleton.get();
        Set<RestoreRange> overlappingRanges = managerGroup.discardOverlappingRanges(instance, restoreJob, otherRanges);
        this.calculateRemainingRangesAndResubmit(restoreJob, instanceId, otherRanges, overlappingRanges);
    }

    private void calculateRemainingRangesAndResubmit(RestoreJob restoreJob, int instanceId, Set<TokenRange> otherRanges, Set<RestoreRange> overlappingRanges) {
        Set existingRanges = overlappingRanges.stream().map(RestoreRange::tokenRange).collect(Collectors.toSet());
        TokenRange.SymmetricDiffResult symmetricDiffResult = TokenRange.symmetricDiff(existingRanges, otherRanges);
        Set remainedRanges = symmetricDiffResult.onlyInLeft;
        this.findSlicesOfCassandraNodeAndSubmit(restoreJob, instanceId, remainedRanges);
    }

    private RestoreJobProgressTracker.Status submit(InstanceMetadata instance, RestoreJob job, RestoreRange range) {
        RestoreJobManagerGroup managerGroup = (RestoreJobManagerGroup)this.restoreJobManagerGroupSingleton.get();
        try {
            return managerGroup.trySubmit(instance, range, job);
        }
        catch (RestoreJobFatalException e) {
            LOGGER.error("Restore range failed. startToken={} endToken={} instance={}", new Object[]{range.startToken(), range.endToken(), range.owner().host(), e});
            range.fail(e);
            this.restoreRangeDatabaseAccessor.updateStatus(range);
            return RestoreJobProgressTracker.Status.FAILED;
        }
    }

    private boolean abortJob(RestoreJob job) {
        LOGGER.info("Abort expired job. jobId={} job={}", (Object)job.jobId, (Object)job);
        try {
            this.restoreJobDatabaseAccessor.abort(job.jobId, "Expired");
            return true;
        }
        catch (Exception exception) {
            LOGGER.warn("Exception on aborting job. jobId: " + job.jobId, (Throwable)exception);
            return false;
        }
    }

    private int delta(LocalDate date1, LocalDate date2) {
        return Math.abs(date1.getDaysSinceEpoch() - date2.getDaysSinceEpoch());
    }

    @VisibleForTesting
    boolean hasInflightJobs() {
        return this.inflightJobsCount != 0;
    }

    @VisibleForTesting
    int jobDiscoveryRecencyDays() {
        return this.jobDiscoveryRecencyDays;
    }

    static class RunContext {
        long nowMillis = System.currentTimeMillis();
        LocalDate today = LocalDate.fromMillisSinceEpoch((long)this.nowMillis);
        int earliestInDays = 0;
        int abortedJobs = 0;
        int expiredJobs = 0;

        RunContext() {
        }
    }

    static class JobIdsByDay {
        private final Map<Integer, Map<UUID, RestoreJobStatus>> jobsByDay = new HashMap<Integer, Map<UUID, RestoreJobStatus>>();
        private final Map<Integer, Set<UUID>> sliceDiscoveredJobsByDay = new HashMap<Integer, Set<UUID>>();
        private final Set<Integer> discoveredDays = new HashSet<Integer>();

        JobIdsByDay() {
        }

        boolean shouldLogJob(RestoreJob job) {
            int day = this.populateDiscoveredDay(job);
            Map jobs = this.jobsByDay.computeIfAbsent(day, key -> new HashMap());
            RestoreJobStatus oldStatus = jobs.put(job.jobId, job.status);
            return oldStatus == null || job.status == RestoreJobStatus.CREATED || oldStatus != job.status;
        }

        void markSlicesDiscovered(RestoreJob job) {
            int day = this.populateDiscoveredDay(job);
            this.sliceDiscoveredJobsByDay.compute(day, (key, value) -> {
                if (value == null) {
                    value = new HashSet<UUID>();
                }
                value.add(job.jobId);
                return value;
            });
        }

        boolean isSliceDiscovered(RestoreJob job) {
            int day = this.populateDiscoveredDay(job);
            return this.sliceDiscoveredJobsByDay.getOrDefault(day, Collections.emptySet()).contains(job.jobId);
        }

        void unsetSlicesDiscovered(RestoreJob job) {
            int day = this.populateDiscoveredDay(job);
            this.sliceDiscoveredJobsByDay.compute(day, (key, value) -> {
                if (value == null) {
                    return null;
                }
                value.remove(job.jobId);
                return value;
            });
        }

        void cleanupMaybe() {
            this.jobsByDay.keySet().removeIf(day -> !this.discoveredDays.contains(day));
            this.discoveredDays.clear();
        }

        private int populateDiscoveredDay(RestoreJob job) {
            int day = job.createdAt.getDaysSinceEpoch();
            this.discoveredDays.add(day);
            return day;
        }

        @VisibleForTesting
        Map<Integer, Map<UUID, RestoreJobStatus>> jobsByDay() {
            return this.jobsByDay;
        }
    }
}

