/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.core.spi.compaction;

import com.google.common.base.Preconditions;
import com.google.gson.Gson;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import org.apache.accumulo.core.client.admin.compaction.CompactableFile;
import org.apache.accumulo.core.conf.ConfigurationTypeHelper;
import org.apache.accumulo.core.conf.Property;
import org.apache.accumulo.core.spi.common.ServiceEnvironment;
import org.apache.accumulo.core.spi.compaction.CompactionExecutorId;
import org.apache.accumulo.core.spi.compaction.CompactionJob;
import org.apache.accumulo.core.spi.compaction.CompactionKind;
import org.apache.accumulo.core.spi.compaction.CompactionPlan;
import org.apache.accumulo.core.spi.compaction.CompactionPlanner;
import org.apache.accumulo.core.util.compaction.CompactionJobPrioritizer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultCompactionPlanner
implements CompactionPlanner {
    private static final Logger log = LoggerFactory.getLogger(DefaultCompactionPlanner.class);
    private List<Executor> executors;
    private int maxFilesToCompact;

    @Override
    @SuppressFBWarnings(value={"UWF_UNWRITTEN_FIELD", "NP_UNWRITTEN_FIELD"}, justification="Field is written by Gson")
    public void init(CompactionPlanner.InitParameters params) {
        if (params.getOptions().containsKey("executors") && !params.getOptions().get("executors").isBlank()) {
            ExecutorConfig[] execConfigs = (ExecutorConfig[])new Gson().fromJson(params.getOptions().get("executors"), ExecutorConfig[].class);
            ArrayList<Executor> tmpExec = new ArrayList<Executor>();
            for (ExecutorConfig executorConfig : execConfigs) {
                CompactionExecutorId ceid;
                Long maxSize;
                Long l = maxSize = executorConfig.maxSize == null ? null : Long.valueOf(ConfigurationTypeHelper.getFixedMemoryAsBytes(executorConfig.maxSize));
                if (executorConfig.type == null) {
                    executorConfig.type = "internal";
                }
                switch (executorConfig.type) {
                    case "internal": {
                        Preconditions.checkArgument((null == executorConfig.queue ? 1 : 0) != 0, (Object)"'queue' should not be specified for internal compactions");
                        int numThreads = Objects.requireNonNull(executorConfig.numThreads, "'numThreads' must be specified for internal type");
                        ceid = params.getExecutorManager().createExecutor(executorConfig.name, numThreads);
                        break;
                    }
                    case "external": {
                        Preconditions.checkArgument((null == executorConfig.numThreads ? 1 : 0) != 0, (Object)"'numThreads' should not be specified for external compactions");
                        String queue = Objects.requireNonNull(executorConfig.queue, "'queue' must be specified for external type");
                        ceid = params.getExecutorManager().getExternalExecutor(queue);
                        break;
                    }
                    default: {
                        throw new IllegalArgumentException("type must be 'internal' or 'external'");
                    }
                }
                tmpExec.add(new Executor(ceid, maxSize));
            }
            Collections.sort(tmpExec, Comparator.comparing(Executor::getMaxSize, Comparator.nullsLast(Comparator.naturalOrder())));
            this.executors = List.copyOf(tmpExec);
            if (this.executors.stream().filter(e -> e.getMaxSize() == null).count() > 1L) {
                throw new IllegalArgumentException("Can only have one executor w/o a maxSize. " + params.getOptions().get("executors"));
            }
        } else {
            throw new IllegalStateException("No defined executors for this planner");
        }
        HashSet maxSizes = new HashSet();
        this.executors.forEach(e -> {
            if (!maxSizes.add(e.getMaxSize())) {
                throw new IllegalArgumentException("Duplicate maxSize set in executors. " + params.getOptions().get("executors"));
            }
        });
        this.determineMaxFilesToCompact(params);
    }

    private void determineMaxFilesToCompact(CompactionPlanner.InitParameters params) {
        String fqo = params.getFullyQualifiedOption("maxOpen");
        if (!params.getServiceEnvironment().getConfiguration().isSet(fqo) && params.getServiceEnvironment().getConfiguration().isSet(Property.TSERV_MAJC_THREAD_MAXOPEN.getKey())) {
            log.warn("The property " + Property.TSERV_MAJC_THREAD_MAXOPEN.getKey() + " was set, it is deprecated.  Set the " + fqo + " option instead.");
            this.maxFilesToCompact = Integer.parseInt(params.getServiceEnvironment().getConfiguration().get(Property.TSERV_MAJC_THREAD_MAXOPEN.getKey()));
        } else {
            this.maxFilesToCompact = Integer.parseInt(params.getOptions().getOrDefault("maxOpen", Property.TSERV_COMPACTION_SERVICE_DEFAULT_MAX_OPEN.getDefaultValue()));
        }
    }

    @Override
    public CompactionPlan makePlan(CompactionPlanner.PlanningParameters params) {
        Collection<Object> group;
        if (params.getCandidates().isEmpty()) {
            return params.createPlanBuilder().build();
        }
        HashSet<CompactableFile> filesCopy = new HashSet<CompactableFile>(params.getCandidates());
        long maxSizeToCompact = this.getMaxSizeToCompact(params.getKind());
        if (params.getRunningCompactions().isEmpty()) {
            group = DefaultCompactionPlanner.findDataFilesToCompact(filesCopy, params.getRatio(), this.maxFilesToCompact, maxSizeToCompact);
            if (!(group.isEmpty() || group.size() >= params.getCandidates().size() || params.getCandidates().size() > this.maxFilesToCompact || params.getKind() != CompactionKind.USER && params.getKind() != CompactionKind.SELECTOR)) {
                filesCopy.removeAll(group);
                filesCopy.add(this.getExpected(group, 0));
                if (DefaultCompactionPlanner.findDataFilesToCompact(filesCopy, params.getRatio(), this.maxFilesToCompact, maxSizeToCompact).isEmpty()) {
                    group = Set.copyOf(params.getCandidates());
                }
            }
        } else if (params.getKind() == CompactionKind.SYSTEM) {
            Set<CompactableFile> expectedFiles = this.getExpected(params.getRunningCompactions());
            if (!Collections.disjoint(filesCopy, expectedFiles)) {
                throw new AssertionError();
            }
            filesCopy.addAll(expectedFiles);
            group = DefaultCompactionPlanner.findDataFilesToCompact(filesCopy, params.getRatio(), this.maxFilesToCompact, maxSizeToCompact);
            if (!Collections.disjoint(group, expectedFiles)) {
                group = Set.of();
            }
        } else {
            group = Set.of();
        }
        if (group.isEmpty()) {
            if ((params.getKind() == CompactionKind.USER || params.getKind() == CompactionKind.SELECTOR || params.getKind() == CompactionKind.CHOP) && params.getRunningCompactions().stream().noneMatch(job -> job.getKind() == params.getKind())) {
                group = DefaultCompactionPlanner.findMaximalRequiredSetToCompact(params.getCandidates(), this.maxFilesToCompact);
            } else if (params.getKind() == CompactionKind.SYSTEM && params.getRunningCompactions().isEmpty() && params.getAll().size() == params.getCandidates().size()) {
                int maxTabletFiles = DefaultCompactionPlanner.getMaxTabletFiles(params.getServiceEnvironment().getConfiguration(params.getTableId()));
                if (params.getAll().size() > maxTabletFiles) {
                    group = this.findFilesToCompactWithLowerRatio(params, maxSizeToCompact, maxTabletFiles);
                }
            }
        }
        if (group.isEmpty()) {
            return params.createPlanBuilder().build();
        }
        CompactionExecutorId ceid = this.getExecutor(group);
        return params.createPlanBuilder().addJob(DefaultCompactionPlanner.createPriority(params, group), ceid, group).build();
    }

    static int getMaxTabletFiles(ServiceEnvironment.Configuration configuration) {
        int maxTabletFiles = Integer.parseInt(configuration.get(Property.TABLE_FILE_MAX.getKey()));
        if (maxTabletFiles <= 0) {
            maxTabletFiles = Integer.parseInt(configuration.get(Property.TSERV_SCAN_MAX_OPENFILES.getKey())) - 1;
        }
        return maxTabletFiles;
    }

    private Collection<CompactableFile> findFilesToCompactWithLowerRatio(CompactionPlanner.PlanningParameters params, long maxSizeToCompact, int maxTabletFiles) {
        double lowRatio = 1.0;
        double highRatio = params.getRatio();
        Preconditions.checkArgument((highRatio >= lowRatio ? 1 : 0) != 0);
        Set<CompactableFile> candidates = Set.copyOf(params.getCandidates());
        Collection<CompactableFile> found = Set.of();
        int goalCompactionSize = candidates.size() - maxTabletFiles + 1;
        if (goalCompactionSize > this.maxFilesToCompact) {
            goalCompactionSize = 0;
        }
        while (highRatio - lowRatio > 0.1) {
            double ratioToCheck = (highRatio - lowRatio) / 2.0 + lowRatio;
            Collection<CompactableFile> filesToCompact = DefaultCompactionPlanner.findDataFilesToCompact(candidates, ratioToCheck, this.maxFilesToCompact, maxSizeToCompact);
            log.trace("Tried ratio {} and found {} {} {}", new Object[]{ratioToCheck, filesToCompact, filesToCompact.size() >= goalCompactionSize, goalCompactionSize});
            if (filesToCompact.isEmpty() || filesToCompact.size() < goalCompactionSize) {
                highRatio = ratioToCheck;
                continue;
            }
            lowRatio = ratioToCheck;
            found = filesToCompact;
        }
        if (found.isEmpty() && lowRatio == 1.0) {
            log.warn("Attempted to lower compaction ration from {} to {} for {} because there are {} files and the max tablet files is {}, however no set of files to compact were found.", new Object[]{params.getRatio(), highRatio, params.getTableId(), params.getCandidates().size(), maxTabletFiles});
        }
        log.info("For {} found {} files to compact lowering compaction ratio from {} to {} because the tablet exceeded {} files, it had {}", new Object[]{params.getTableId(), found.size(), params.getRatio(), lowRatio, maxTabletFiles, params.getCandidates().size()});
        return found;
    }

    private static short createPriority(CompactionPlanner.PlanningParameters params, Collection<CompactableFile> group) {
        return CompactionJobPrioritizer.createPriority(params.getKind(), params.getAll().size(), group.size());
    }

    private long getMaxSizeToCompact(CompactionKind kind) {
        Long max;
        if (kind == CompactionKind.SYSTEM && (max = this.executors.get((int)(this.executors.size() - 1)).maxSize) != null) {
            return max;
        }
        return Long.MAX_VALUE;
    }

    private CompactableFile getExpected(Collection<CompactableFile> files, int count) {
        long size = files.stream().mapToLong(CompactableFile::getEstimatedSize).sum();
        try {
            return CompactableFile.create(new URI("hdfs://fake/accumulo/tables/adef/t-zzFAKEzz/FAKE-0000" + count + ".rf"), size, 0L);
        }
        catch (URISyntaxException e) {
            throw new RuntimeException(e);
        }
    }

    private Set<CompactableFile> getExpected(Collection<CompactionJob> compacting) {
        HashSet<CompactableFile> expected = new HashSet<CompactableFile>();
        int count = 0;
        for (CompactionJob job : compacting) {
            expected.add(this.getExpected(job.getFiles(), ++count));
        }
        return expected;
    }

    private static Collection<CompactableFile> findMaximalRequiredSetToCompact(Collection<CompactableFile> files, int maxFilesToCompact) {
        if (files.size() <= maxFilesToCompact) {
            return files;
        }
        List<CompactableFile> sortedFiles = DefaultCompactionPlanner.sortByFileSize(files);
        int numToCompact = maxFilesToCompact;
        if (sortedFiles.size() > maxFilesToCompact && sortedFiles.size() < 2 * maxFilesToCompact) {
            numToCompact = sortedFiles.size() - maxFilesToCompact + 1;
        }
        return sortedFiles.subList(0, numToCompact);
    }

    static Collection<CompactableFile> findDataFilesToCompact(Set<CompactableFile> files, double ratio, int maxFilesToCompact, long maxSizeToCompact) {
        if (files.size() <= 1) {
            return Collections.emptySet();
        }
        List<CompactableFile> sortedFiles = DefaultCompactionPlanner.sortByFileSize(files);
        int maxSizeIndex = sortedFiles.size();
        long sum = 0L;
        for (int i = 0; i < sortedFiles.size(); ++i) {
            if ((sum += sortedFiles.get(i).getEstimatedSize()) <= maxSizeToCompact) continue;
            maxSizeIndex = i;
            break;
        }
        if (maxSizeIndex < sortedFiles.size() && (sortedFiles = sortedFiles.subList(0, maxSizeIndex)).size() <= 1) {
            return Collections.emptySet();
        }
        int windowStart = 0;
        for (int windowEnd = Math.min(sortedFiles.size(), maxFilesToCompact); windowEnd <= sortedFiles.size(); ++windowEnd) {
            Collection<CompactableFile> filesToCompact = DefaultCompactionPlanner.findDataFilesToCompact(sortedFiles.subList(windowStart, windowEnd), ratio);
            if (!filesToCompact.isEmpty()) {
                return filesToCompact;
            }
            ++windowStart;
        }
        return Collections.emptySet();
    }

    private static Collection<CompactableFile> findDataFilesToCompact(List<CompactableFile> sortedFiles, double ratio) {
        int larsmaIndex = -1;
        long larsmaSum = Long.MIN_VALUE;
        int goodIndex = -1;
        long sum = sortedFiles.get(0).getEstimatedSize();
        for (int c = 1; c < sortedFiles.size(); ++c) {
            long currSize = sortedFiles.get(c).getEstimatedSize();
            Preconditions.checkArgument((currSize >= sortedFiles.get(c - 1).getEstimatedSize() ? 1 : 0) != 0);
            sum += currSize;
            if ((double)currSize * ratio < (double)sum) {
                goodIndex = c;
                continue;
            }
            if (c - 1 != goodIndex) continue;
            if (larsmaIndex != -1 && larsmaSum <= sortedFiles.get(goodIndex).getEstimatedSize()) break;
            larsmaIndex = goodIndex;
            larsmaSum = sum - currSize;
        }
        if (sortedFiles.size() - 1 == goodIndex && (larsmaIndex == -1 || larsmaSum > sortedFiles.get(goodIndex).getEstimatedSize())) {
            larsmaIndex = goodIndex;
        }
        if (larsmaIndex == -1) {
            return Collections.emptySet();
        }
        return sortedFiles.subList(0, larsmaIndex + 1);
    }

    CompactionExecutorId getExecutor(Collection<CompactableFile> files) {
        long size = files.stream().mapToLong(CompactableFile::getEstimatedSize).sum();
        for (Executor executor : this.executors) {
            if (executor.maxSize != null && size >= executor.maxSize) continue;
            return executor.ceid;
        }
        return this.executors.get((int)(this.executors.size() - 1)).ceid;
    }

    private static List<CompactableFile> sortByFileSize(Collection<CompactableFile> files) {
        ArrayList<CompactableFile> sortedFiles = new ArrayList<CompactableFile>(files);
        Collections.sort(sortedFiles, Comparator.comparingLong(CompactableFile::getEstimatedSize).thenComparing(CompactableFile::getUri));
        return sortedFiles;
    }

    private static class ExecutorConfig {
        String type;
        String name;
        String maxSize;
        Integer numThreads;
        String queue;

        private ExecutorConfig() {
        }
    }

    private static class Executor {
        final CompactionExecutorId ceid;
        final Long maxSize;

        public Executor(CompactionExecutorId ceid, Long maxSize) {
            Preconditions.checkArgument((maxSize == null || maxSize > 0L ? 1 : 0) != 0, (Object)"Invalid value for maxSize");
            this.ceid = Objects.requireNonNull(ceid, "Compaction ID is null");
            this.maxSize = maxSize;
        }

        Long getMaxSize() {
            return this.maxSize;
        }

        public String toString() {
            return "[ceid=" + this.ceid + ", maxSize=" + this.maxSize + "]";
        }
    }
}

