"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.callTokenEndpoint = callTokenEndpoint;
exports.composeLogoutUrl = composeLogoutUrl;
exports.getBaseRedirectUrl = getBaseRedirectUrl;
exports.getExpirationDate = getExpirationDate;
exports.getNextUrl = getNextUrl;
exports.getRootUrl = getRootUrl;
exports.parseTokenResponse = parseTokenResponse;

var _querystring = require("querystring");

/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
function parseTokenResponse(payload) {
  const payloadString = payload.toString();

  if (payloadString.trim()[0] === '{') {
    try {
      return JSON.parse(payloadString);
    } catch (error) {
      throw Error(`Invalid JSON payload: ${error}`);
    }
  }

  return (0, _querystring.parse)(payloadString);
}

function getRootUrl(config, core, request) {
  var _config$openid;

  const host = core.http.getServerInfo().hostname;
  const port = core.http.getServerInfo().port;
  let protocol = core.http.getServerInfo().protocol;
  let httpHost = `${host}:${port}`;

  if ((_config$openid = config.openid) !== null && _config$openid !== void 0 && _config$openid.trust_dynamic_headers) {
    const xForwardedHost = request.headers['x-forwarded-host'] || undefined;
    const xForwardedProto = request.headers['x-forwarded-proto'] || undefined;

    if (xForwardedHost) {
      httpHost = xForwardedHost;
    }

    if (xForwardedProto) {
      protocol = xForwardedProto;
    }
  }

  return `${protocol}://${httpHost}`;
}

function getBaseRedirectUrl(config, core, request) {
  var _config$openid2;

  if ((_config$openid2 = config.openid) !== null && _config$openid2 !== void 0 && _config$openid2.base_redirect_url) {
    const baseRedirectUrl = config.openid.base_redirect_url;
    return baseRedirectUrl.endsWith('/') ? baseRedirectUrl.slice(0, -1) : baseRedirectUrl;
  }

  const rootUrl = getRootUrl(config, core, request);

  if (core.http.basePath.serverBasePath) {
    return `${rootUrl}${core.http.basePath.serverBasePath}`;
  }

  return rootUrl;
}

function getNextUrl(config, core, request) {
  return request.query.nextUrl || getBaseRedirectUrl(config, core, request) || '/';
}

async function callTokenEndpoint(tokenEndpoint, query, wreckClient) {
  var _tokenResponse$res;

  const tokenResponse = await wreckClient.post(tokenEndpoint, {
    payload: (0, _querystring.stringify)(query),
    headers: {
      'Content-Type': 'application/x-www-form-urlencoded'
    }
  });

  if (!((_tokenResponse$res = tokenResponse.res) !== null && _tokenResponse$res !== void 0 && _tokenResponse$res.statusCode) || tokenResponse.res.statusCode < 200 || tokenResponse.res.statusCode > 299) {
    throw new Error(`Failed calling token endpoint: ${tokenResponse.res.statusCode} ${tokenResponse.res.statusMessage}`);
  }

  const tokenPayload = parseTokenResponse(tokenResponse.payload);
  return {
    idToken: tokenPayload.id_token,
    accessToken: tokenPayload.access_token,
    refreshToken: tokenPayload.refresh_token,
    expiresIn: tokenPayload.expires_in
  };
}

function composeLogoutUrl(customLogoutUrl, idpEndsessionEndpoint, additionalQueryParams) {
  const logoutEndpont = customLogoutUrl || idpEndsessionEndpoint;
  const logoutUrl = new URL(logoutEndpont);
  Object.keys(additionalQueryParams).forEach(key => {
    logoutUrl.searchParams.append(key, additionalQueryParams[key]);
  });
  return logoutUrl.toString();
}

function getExpirationDate(tokenResponse) {
  if (!tokenResponse) {
    throw new Error('Invalid token');
  } else if (tokenResponse.idToken) {
    const idToken = tokenResponse.idToken;
    const parts = idToken.split('.');

    if (parts.length !== 3) {
      throw new Error('Invalid token');
    }

    const claim = JSON.parse(Buffer.from(parts[1], 'base64').toString());
    return claim.exp * 1000;
  } else {
    return Date.now() + tokenResponse.expiresIn * 1000;
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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