"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createSavedSearchReport = createSavedSearchReport;

var _dataReportHelpers = require("./dataReportHelpers");

var _helpers = require("./helpers");

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

/**
 * Specify how long scroll context should be maintained for scrolled search
 */
const scrollTimeout = '1m';

async function createSavedSearchReport(report, client, dateFormat, csvSeparator, allowLeadingWildcards, isScheduledTask = true, logger) {
  const params = report.report_definition.report_params;
  const reportFormat = params.core_params.report_format;
  const reportName = params.report_name;
  await populateMetaData(client, report, isScheduledTask, logger);
  const data = await generateReportData(client, params.core_params, dateFormat, csvSeparator, allowLeadingWildcards, isScheduledTask, logger);
  const curTime = new Date();
  const timeCreated = curTime.valueOf();
  const fileName = (0, _helpers.getFileName)(reportName, curTime) + '.' + reportFormat;
  return {
    timeCreated,
    dataUrl: data,
    fileName
  };
}
/**
 * Populate parameters and saved search info related to meta data object.
 * @param client  OpenSearch client
 * @param report  Report input
 */


async function populateMetaData(client, report, isScheduledTask, logger) {
  _dataReportHelpers.metaData.saved_search_id = report.report_definition.report_params.core_params.saved_search_id;
  _dataReportHelpers.metaData.report_format = report.report_definition.report_params.core_params.report_format;
  _dataReportHelpers.metaData.start = report.time_from;
  _dataReportHelpers.metaData.end = report.time_to; // Get saved search info

  let resIndexPattern = {};
  const ssParams = {
    index: '.kibana',
    id: 'search:' + _dataReportHelpers.metaData.saved_search_id
  };
  const ssInfos = await (0, _helpers.callCluster)(client, 'get', ssParams, isScheduledTask);
  _dataReportHelpers.metaData.sorting = ssInfos._source.search.sort;
  _dataReportHelpers.metaData.type = ssInfos._source.type;
  _dataReportHelpers.metaData.searchSourceJSON = ssInfos._source.search.kibanaSavedObjectMeta.searchSourceJSON; // Get the list of selected columns in the saved search.Otherwise select all the fields under the _source

  await (0, _dataReportHelpers.getSelectedFields)(ssInfos._source.search.columns); // Get index name

  for (const item of ssInfos._source.references) {
    if (item.name === JSON.parse(_dataReportHelpers.metaData.searchSourceJSON).indexRefName) {
      // Get index-pattern information
      const indexPattern = await (0, _helpers.callCluster)(client, 'get', {
        index: '.kibana',
        id: 'index-pattern:' + item.id
      }, isScheduledTask);
      resIndexPattern = indexPattern._source['index-pattern'];
      _dataReportHelpers.metaData.paternName = resIndexPattern.title;
      _dataReportHelpers.metaData.timeFieldName = resIndexPattern.timeFieldName, _dataReportHelpers.metaData.fields = resIndexPattern.fields; // Get all fields
      // Getting fields of type Date

      const dateFields = [];

      for (const item of JSON.parse(_dataReportHelpers.metaData.fields)) {
        if (item.type === 'date') {
          dateFields.push(item.name);
        }
      }

      _dataReportHelpers.metaData.dateFields = dateFields;
    }
  }
}
/**
 * Generate CSV data by query and convert OpenSearch data set.
 * @param client  OpenSearch client
 * @param limit   limit size of result data set
 */


async function generateReportData(client, params, dateFormat, csvSeparator, allowLeadingWildcards, isScheduledTask, logger) {
  let opensearchData = {};
  const arrayHits = [];
  const report = {
    _source: _dataReportHelpers.metaData
  };
  const indexPattern = report._source.paternName;
  const maxResultSize = await getMaxResultSize();
  const opensearchCount = await getOpenSearchDataSize();
  const total = Math.min(opensearchCount.count, params.limit);

  if (total === 0) {
    return '';
  }

  const reqBody = (0, _dataReportHelpers.buildRequestBody)(report, allowLeadingWildcards, 0);
  logger.info(`[Reporting csv module] DSL request body: ${JSON.stringify(reqBody)}`);

  if (total > maxResultSize) {
    await getOpenSearchDataByScroll();
  } else {
    await getOpenSearchDataBySearch();
  }

  return convertOpenSearchDataToCsv(); // Fetch OpenSearch query max size windows to decide search or scroll

  async function getMaxResultSize() {
    const settings = await (0, _helpers.callCluster)(client, 'indices.getSettings', {
      index: indexPattern,
      includeDefaults: true
    }, isScheduledTask);
    let maxResultSize = Number.MAX_SAFE_INTEGER;

    for (let indexName in settings) {
      // The location of max result window differs if default overridden.
      maxResultSize = Math.min(maxResultSize, settings[indexName].settings.index.max_result_window || settings[indexName].defaults.index.max_result_window);
    }

    return maxResultSize;
  } // Build the OpenSearch Count query to count the size of result


  async function getOpenSearchDataSize() {
    const countReq = (0, _dataReportHelpers.buildRequestBody)(report, allowLeadingWildcards, 1);
    return await (0, _helpers.callCluster)(client, 'count', {
      index: indexPattern,
      body: countReq
    }, isScheduledTask);
  }

  async function getOpenSearchDataByScroll() {
    const searchParams = {
      index: report._source.paternName,
      scroll: scrollTimeout,
      body: reqBody,
      size: maxResultSize
    }; // Open scroll context by fetching first batch

    opensearchData = await (0, _helpers.callCluster)(client, 'search', searchParams, isScheduledTask);
    arrayHits.push(opensearchData.hits); // Start scrolling till the end

    const nbScroll = Math.floor(total / maxResultSize);

    for (let i = 0; i < nbScroll; i++) {
      const resScroll = await (0, _helpers.callCluster)(client, 'scroll', {
        scrollId: opensearchData._scroll_id,
        scroll: scrollTimeout
      }, isScheduledTask);

      if (Object.keys(resScroll.hits.hits).length > 0) {
        arrayHits.push(resScroll.hits);
      }
    } // Clear scroll context


    await (0, _helpers.callCluster)(client, 'clearScroll', {
      scrollId: opensearchData._scroll_id
    }, isScheduledTask);
  }

  async function getOpenSearchDataBySearch() {
    const searchParams = {
      index: report._source.paternName,
      body: reqBody,
      size: total
    };
    opensearchData = await (0, _helpers.callCluster)(client, 'search', searchParams, isScheduledTask);
    arrayHits.push(opensearchData.hits);
  } // Parse OpenSearch data and convert to CSV


  async function convertOpenSearchDataToCsv() {
    const dataset = [];
    dataset.push((0, _dataReportHelpers.getOpenSearchData)(arrayHits, report, params, dateFormat));
    return await (0, _dataReportHelpers.convertToCSV)(dataset, csvSeparator);
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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