"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.VisualizationsRouter = VisualizationsRouter;

var _configSchema = require("@osd/config-schema");

var _custom_panel_adaptor = require("../../adaptors/custom_panels/custom_panel_adaptor");

var _custom_panels = require("../../../common/constants/custom_panels");

var _metrics_helper = require("../../common/metrics/metrics_helper");

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
function VisualizationsRouter(router) {
  // Fetch all the savedVisualzations
  const customPanelBackend = new _custom_panel_adaptor.CustomPanelsAdaptor();
  router.get({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/visualizations`,
    validate: {}
  }, async (context, request, response) => {
    (0, _metrics_helper.addRequestToMetric)('operational_panels', 'fetch_visualization', 'count');
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const visualizationList = await customPanelBackend.getAllSavedVisualizations(opensearchNotebooksClient);
      return response.ok({
        body: {
          visualizations: visualizationList
        }
      });
    } catch (error) {
      (0, _metrics_helper.addRequestToMetric)('operational_panels', 'fetch_visualization', error);
      console.error('Issue in fetching saved visualizations:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // get all saved visualizations by Id

  router.get({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/visualizations/{savedVisualizationId}`,
    validate: {
      params: _configSchema.schema.object({
        savedVisualizationId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const savedVisualization = await customPanelBackend.getSavedVisualizationById(opensearchNotebooksClient, request.params.savedVisualizationId);
      return response.ok({
        body: {
          visualization: savedVisualization
        }
      });
    } catch (error) {
      console.error('Issue in fetching saved visualizations by ids:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Add a new visualization to the panel

  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/visualizations`,
    validate: {
      body: _configSchema.schema.object({
        panelId: _configSchema.schema.string(),
        savedVisualizationId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const newVisualizations = await customPanelBackend.addVisualization(opensearchNotebooksClient, request.body.panelId, request.body.savedVisualizationId);
      return response.ok({
        body: {
          message: 'Visualization Added',
          visualizations: newVisualizations
        }
      });
    } catch (error) {
      console.error('Issue in adding visualization:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Add multiple visualizations to the panel

  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/visualizations/multiple`,
    validate: {
      body: _configSchema.schema.object({
        panelId: _configSchema.schema.string(),
        savedVisualizationIds: _configSchema.schema.arrayOf(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const allVisualizations = await customPanelBackend.addMultipleVisualizations(opensearchNotebooksClient, request.body.panelId, request.body.savedVisualizationIds);
      return response.ok({
        body: {
          message: 'Visualizations Added',
          visualizations: allVisualizations
        }
      });
    } catch (error) {
      console.error('Issue in adding visualization:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Replace an existing visualization

  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/visualizations/replace`,
    validate: {
      body: _configSchema.schema.object({
        panelId: _configSchema.schema.string(),
        savedVisualizationId: _configSchema.schema.string(),
        oldVisualizationId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const newVisualizations = await customPanelBackend.addVisualization(opensearchNotebooksClient, request.body.panelId, request.body.savedVisualizationId, request.body.oldVisualizationId);
      return response.ok({
        body: {
          message: 'Visualization Replaced',
          visualizations: newVisualizations
        }
      });
    } catch (error) {
      console.error('Issue in replacing visualization:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // changes the position of the mentioned visualizations
  // Also removes the visualizations not mentioned

  router.put({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/visualizations/edit`,
    validate: {
      body: _configSchema.schema.object({
        panelId: _configSchema.schema.string(),
        visualizationParams: _configSchema.schema.arrayOf(_configSchema.schema.object({
          i: _configSchema.schema.string(),
          x: _configSchema.schema.number(),
          y: _configSchema.schema.number(),
          w: _configSchema.schema.number(),
          h: _configSchema.schema.number()
        }))
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const newVisualizations = await customPanelBackend.editVisualization(opensearchNotebooksClient, request.body.panelId, request.body.visualizationParams);
      return response.ok({
        body: {
          message: 'Visualizations Edited',
          visualizations: newVisualizations
        }
      });
    } catch (error) {
      console.error('Issue in Editing visualizations:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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