"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateTemplate = exports.validateInstance = void 0;

var _ajv = _interopRequireDefault(require("ajv"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
const ajv = new _ajv.default();
const staticAsset = {
  type: 'object',
  properties: {
    path: {
      type: 'string'
    },
    annotation: {
      type: 'string',
      nullable: true
    }
  },
  required: ['path'],
  additionalProperties: false
};
const templateSchema = {
  type: 'object',
  properties: {
    name: {
      type: 'string'
    },
    version: {
      type: 'string'
    },
    displayName: {
      type: 'string',
      nullable: true
    },
    license: {
      type: 'string'
    },
    type: {
      type: 'string'
    },
    labels: {
      type: 'array',
      items: {
        type: 'string'
      },
      nullable: true
    },
    author: {
      type: 'string',
      nullable: true
    },
    description: {
      type: 'string',
      nullable: true
    },
    sourceUrl: {
      type: 'string',
      nullable: true
    },
    statics: {
      type: 'object',
      properties: {
        logo: { ...staticAsset,
          nullable: true
        },
        gallery: {
          type: 'array',
          items: staticAsset,
          nullable: true
        },
        darkModeLogo: { ...staticAsset,
          nullable: true
        },
        darkModeGallery: {
          type: 'array',
          items: staticAsset,
          nullable: true
        }
      },
      additionalProperties: false,
      nullable: true
    },
    components: {
      type: 'array',
      items: {
        type: 'object',
        properties: {
          name: {
            type: 'string'
          },
          version: {
            type: 'string'
          }
        },
        required: ['name', 'version']
      }
    },
    assets: {
      type: 'object',
      properties: {
        savedObjects: {
          type: 'object',
          properties: {
            name: {
              type: 'string'
            },
            version: {
              type: 'string'
            }
          },
          required: ['name', 'version'],
          nullable: true,
          additionalProperties: false
        }
      },
      additionalProperties: false
    },
    sampleData: {
      type: 'object',
      properties: {
        path: {
          type: 'string'
        }
      },
      required: ['path'],
      additionalProperties: false,
      nullable: true
    }
  },
  required: ['name', 'version', 'license', 'type', 'components', 'assets'],
  additionalProperties: false
};
const instanceSchema = {
  type: 'object',
  properties: {
    name: {
      type: 'string'
    },
    templateName: {
      type: 'string'
    },
    dataSource: {
      type: 'string'
    },
    creationDate: {
      type: 'string'
    },
    assets: {
      type: 'array',
      items: {
        type: 'object',
        properties: {
          assetType: {
            type: 'string'
          },
          assetId: {
            type: 'string'
          },
          isDefaultAsset: {
            type: 'boolean'
          },
          description: {
            type: 'string'
          }
        },
        required: ['assetType', 'assetId', 'isDefaultAsset', 'description']
      }
    }
  },
  required: ['name', 'templateName', 'dataSource', 'creationDate', 'assets']
};
const templateValidator = ajv.compile(templateSchema);
const instanceValidator = ajv.compile(instanceSchema);
/**
 * Validates an integration template against a predefined schema using the AJV library.
 * Since AJV validators use side effects for errors,
 * this is a more conventional wrapper that simplifies calling.
 *
 * @param data The data to be validated as an IntegrationTemplate.
 * @return A ValidationResult indicating whether the validation was successful or not.
 *         If validation succeeds, returns an object with 'ok' set to true and the validated data.
 *         If validation fails, returns an object with 'ok' set to false and an Error object describing the validation error.
 */

const validateTemplate = data => {
  if (!templateValidator(data)) {
    return {
      ok: false,
      error: new Error(ajv.errorsText(templateValidator.errors))
    };
  } // We assume an invariant that the type of an integration is connected with its component.


  if (data.components.findIndex(x => x.name === data.type) < 0) {
    return {
      ok: false,
      error: new Error(`The integration type '${data.type}' must be included as a component`)
    };
  }

  return {
    ok: true,
    value: data
  };
};
/**
 * Validates an integration instance against a predefined schema using the AJV library.
 *
 * @param data The data to be validated as an IntegrationInstance.
 * @return A ValidationResult indicating whether the validation was successful or not.
 *         If validation succeeds, returns an object with 'ok' set to true and the validated data.
 *         If validation fails, returns an object with 'ok' set to false and an Error object describing the validation error.
 */


exports.validateTemplate = validateTemplate;

const validateInstance = data => {
  if (!instanceValidator(data)) {
    return {
      ok: false,
      error: new Error(ajv.errorsText(instanceValidator.errors))
    };
  }

  return {
    ok: true,
    value: data
  };
};

exports.validateInstance = validateInstance;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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