"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Integration = void 0;

var fs = _interopRequireWildcard(require("fs/promises"));

var _path = _interopRequireDefault(require("path"));

var _validators = require("../validators");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }

function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

/**
 * Helper function to compare version numbers.
 * Assumes that the version numbers are valid, produces undefined behavior otherwise.
 *
 * @param a Left-hand number
 * @param b Right-hand number
 * @returns -1 if a > b, 1 if a < b, 0 otherwise.
 */
function compareVersions(a, b) {
  const aParts = a.split('.').map(Number.parseInt);
  const bParts = b.split('.').map(Number.parseInt);

  for (let i = 0; i < Math.max(aParts.length, bParts.length); i++) {
    const aValue = i < aParts.length ? aParts[i] : 0;
    const bValue = i < bParts.length ? bParts[i] : 0;

    if (aValue > bValue) {
      return -1; // a > b
    } else if (aValue < bValue) {
      return 1; // a < b
    }
  }

  return 0; // a == b
}
/**
 * Helper function to check if the given path is a directory
 *
 * @param dirPath The directory to check.
 * @returns True if the path is a directory.
 */


async function isDirectory(dirPath) {
  try {
    const stats = await fs.stat(dirPath);
    return stats.isDirectory();
  } catch {
    return false;
  }
}
/**
 * The Integration class represents the data for Integration Templates.
 * It is backed by the repository file system.
 * It includes accessor methods for integration configs, as well as helpers for nested components.
 */


class Integration {
  constructor(directory) {
    _defineProperty(this, "directory", void 0);

    _defineProperty(this, "name", void 0);

    this.directory = directory;
    this.name = _path.default.basename(directory);
  }
  /**
   * Check the integration for validity.
   * This is not a deep check, but a quick check to verify that the integration is a valid directory and has a config file.
   *
   * @returns true if the integration is valid.
   */


  async check() {
    if (!(await isDirectory(this.directory))) {
      return false;
    }

    return (await this.getConfig()) !== null;
  }
  /**
   * Like check(), but thoroughly checks all nested integration dependencies.
   *
   * @returns true if the integration is valid.
   */


  async deepCheck() {
    if (!(await this.check())) {
      console.error('check failed');
      return false;
    }

    try {
      // An integration must have at least one mapping
      const schemas = await this.getSchemas();

      if (Object.keys(schemas.mappings).length === 0) {
        return false;
      } // An integration must have at least one asset


      const assets = await this.getAssets();

      if (Object.keys(assets).length === 0) {
        return false;
      }
    } catch (err) {
      // Any loading errors are considered invalid
      console.error('Deep check failed for exception', err);
      return false;
    }

    return true;
  }
  /**
   * Get the latest version of the integration available.
   * This method relies on the fact that integration configs have their versions in their name.
   * Any files that don't match the config naming convention will be ignored.
   *
   * @returns A string with the latest version, or null if no versions are available.
   */


  async getLatestVersion() {
    const files = await fs.readdir(this.directory);
    const versions = [];

    for (const file of files) {
      if (_path.default.extname(file) === '.json' && file.startsWith(`${this.name}-`)) {
        const version = file.substring(this.name.length + 1, file.length - 5);

        if (!version.match(/^\d+(\.\d+)*$/)) {
          continue;
        }

        versions.push(version);
      }
    }

    versions.sort((a, b) => compareVersions(a, b));
    return versions.length > 0 ? versions[0] : null;
  }
  /**
   * Get the configuration of the current integration.
   *
   * @param version The version of the config to retrieve.
   * @returns The config if a valid config matching the version is present, otherwise null.
   */


  async getConfig(version) {
    const maybeVersion = version ? version : await this.getLatestVersion();

    if (maybeVersion === null) {
      return null;
    }

    const configFile = `${this.name}-${maybeVersion}.json`;

    const configPath = _path.default.join(this.directory, configFile);

    try {
      const config = await fs.readFile(configPath, {
        encoding: 'utf-8'
      });
      const possibleTemplate = JSON.parse(config);
      const template = (0, _validators.validateTemplate)(possibleTemplate);

      if (template.ok) {
        return template.value;
      }

      console.error(template.error);
      return null;
    } catch (err) {
      if (err instanceof SyntaxError) {
        console.error(`Syntax errors in ${configFile}`, err);
        return null;
      }

      if (err instanceof Error && err.code === 'ENOENT') {
        console.error(`Attempted to retrieve non-existent config ${configFile}`);
        return null;
      }

      throw new Error('Could not load integration', {
        cause: err
      });
    }
  }
  /**
   * Retrieve assets associated with the integration.
   * This method greedily retrieves all assets.
   * If the version is invalid, an error is thrown.
   * If an asset is invalid, it will be skipped.
   *
   * @param version The version of the integration to retrieve assets for.
   * @returns An object containing the different types of assets.
   */


  async getAssets(version) {
    const config = await this.getConfig(version);

    if (config === null) {
      return Promise.reject(new Error('Attempted to get assets of invalid config'));
    }

    const result = {};

    if (config.assets.savedObjects) {
      const sobjPath = _path.default.join(this.directory, 'assets', `${config.assets.savedObjects.name}-${config.assets.savedObjects.version}.ndjson`);

      try {
        const ndjson = await fs.readFile(sobjPath, {
          encoding: 'utf-8'
        });
        const asJson = '[' + ndjson.trim().replace(/\n/g, ',') + ']';
        const parsed = JSON.parse(asJson);
        result.savedObjects = parsed;
      } catch (err) {
        console.error("Failed to load saved object assets, proceeding as if it's absent", err);
      }
    }

    return result;
  }
  /**
   * Retrieve sample data associated with the integration.
   * If the version is invalid, an error is thrown.
   * If the sample data is invalid, null will be returned
   *
   * @param version The version of the integration to retrieve assets for.
   * @returns An object containing a list of sample data with adjusted timestamps.
   */


  async getSampleData(version) {
    const config = await this.getConfig(version);

    if (config === null) {
      return Promise.reject(new Error('Attempted to get assets of invalid config'));
    }

    const result = {
      sampleData: null
    };

    if (config.sampleData) {
      var _config$sampleData;

      const sobjPath = _path.default.join(this.directory, 'data', (_config$sampleData = config.sampleData) === null || _config$sampleData === void 0 ? void 0 : _config$sampleData.path);

      try {
        const jsonContent = await fs.readFile(sobjPath, {
          encoding: 'utf-8'
        });
        const parsed = JSON.parse(jsonContent);

        for (const value of parsed) {
          if (!('@timestamp' in value)) {
            continue;
          } // Randomly scatter timestamps across last 10 minutes
          // Assume for now that the ordering of events isn't important, can change to a sequence if needed
          // Also doesn't handle fields like `observedTimestamp` if present


          Object.assign(value, {
            '@timestamp': new Date(Date.now() - Math.floor(Math.random() * 1000 * 60 * 10)).toISOString()
          });
        }

        result.sampleData = parsed;
      } catch (err) {
        console.error("Failed to load saved object assets, proceeding as if it's absent", err);
      }
    }

    return result;
  }
  /**
   * Retrieve schema data associated with the integration.
   * This method greedily retrieves all mappings and schemas.
   * It's assumed that a valid version will be provided.
   * If the version is invalid, an error is thrown.
   * If a schema is invalid, an error will be thrown.
   *
   * @param version The version of the integration to retrieve assets for.
   * @returns An object containing the different types of assets.
   */


  async getSchemas(version) {
    const config = await this.getConfig(version);

    if (config === null) {
      return Promise.reject(new Error('Attempted to get assets of invalid config'));
    }

    const result = {
      mappings: {}
    };

    try {
      for (const component of config.components) {
        const schemaFile = `${component.name}-${component.version}.mapping.json`;
        const rawSchema = await fs.readFile(_path.default.join(this.directory, 'schemas', schemaFile), {
          encoding: 'utf-8'
        });
        const parsedSchema = JSON.parse(rawSchema);
        result.mappings[component.name] = parsedSchema;
      }
    } catch (err) {
      // It's not clear that an invalid schema can be recovered from.
      // For integrations to function, we need schemas to be valid.
      console.error('Error loading schema', err);
      return Promise.reject(new Error('Could not load schema', {
        cause: err
      }));
    }

    return result;
  }
  /**
   * Retrieves the data for a static file associated with the integration.
   *
   * @param staticPath The path of the static to retrieve.
   * @returns A buffer with the static's data if present, otherwise null.
   */


  async getStatic(staticPath) {
    const fullStaticPath = _path.default.join(this.directory, 'static', staticPath);

    try {
      return await fs.readFile(fullStaticPath);
    } catch (err) {
      if (err instanceof Error && err.code === 'ENOENT') {
        console.error(`Static not found: ${staticPath}`);
        return null;
      }

      throw err;
    }
  }

}

exports.Integration = Integration;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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