"use strict";
/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Any modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ensureByteSizeValue = exports.tb = exports.gb = exports.mb = exports.kb = exports.bytes = exports.ByteSizeValue = void 0;
const unitMultiplier = {
    b: Math.pow(1024, 0),
    gb: Math.pow(1024, 3),
    kb: Math.pow(1024, 1),
    mb: Math.pow(1024, 2),
};
function renderUnit(value, unit) {
    const prettyValue = Number(value.toFixed(2));
    return `${prettyValue}${unit}`;
}
class ByteSizeValue {
    constructor(valueInBytes) {
        this.valueInBytes = valueInBytes;
        if (!Number.isSafeInteger(valueInBytes) || valueInBytes < 0) {
            throw new Error(`Value in bytes is expected to be a safe positive integer.`);
        }
    }
    static parse(text) {
        const match = /([1-9][0-9]*)(b|kb|mb|gb)/.exec(text);
        if (!match) {
            const number = Number(text);
            if (typeof number !== 'number' || isNaN(number)) {
                throw new Error(`Failed to parse value as byte value. Value must be either number of bytes, or follow the format <count>[b|kb|mb|gb] ` +
                    `(e.g., '1024kb', '200mb', '1gb'), where the number is a safe positive integer.`);
            }
            return new ByteSizeValue(number);
        }
        const value = parseInt(match[1], 0);
        const unit = match[2];
        return new ByteSizeValue(value * unitMultiplier[unit]);
    }
    isGreaterThan(other) {
        return this.valueInBytes > other.valueInBytes;
    }
    isLessThan(other) {
        return this.valueInBytes < other.valueInBytes;
    }
    isEqualTo(other) {
        return this.valueInBytes === other.valueInBytes;
    }
    getValueInBytes() {
        return this.valueInBytes;
    }
    toString(returnUnit) {
        let value = this.valueInBytes;
        let unit = `b`;
        for (const nextUnit of ['kb', 'mb', 'gb']) {
            if (unit === returnUnit || (returnUnit == null && value < 1024)) {
                return renderUnit(value, unit);
            }
            value = value / 1024;
            unit = nextUnit;
        }
        return renderUnit(value, unit);
    }
}
exports.ByteSizeValue = ByteSizeValue;
exports.bytes = (value) => new ByteSizeValue(value);
exports.kb = (value) => exports.bytes(value * 1024);
exports.mb = (value) => exports.kb(value * 1024);
exports.gb = (value) => exports.mb(value * 1024);
exports.tb = (value) => exports.gb(value * 1024);
function ensureByteSizeValue(value) {
    if (typeof value === 'string') {
        return ByteSizeValue.parse(value);
    }
    if (typeof value === 'number') {
        return new ByteSizeValue(value);
    }
    return value;
}
exports.ensureByteSizeValue = ensureByteSizeValue;
