// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
// @ts-ignore
import { Transform as PortableTransform } from 'readable-stream';
import { NodeAlgorithmSuite, getDecryptionHelper, CommitmentPolicy, CommitmentPolicySuites, SignaturePolicy, SignaturePolicySuites, needs, } from '@aws-crypto/material-management-node';
import { deserializeFactory } from '@aws-crypto/serialize';
const toUtf8 = (input) => Buffer.from(input.buffer, input.byteOffset, input.byteLength).toString('utf8');
const deserialize = deserializeFactory(toUtf8, NodeAlgorithmSuite);
const PortableTransformWithType = PortableTransform;
export class ParseHeaderStream extends PortableTransformWithType {
    materialsManager;
    commitmentPolicy;
    signaturePolicy;
    maxEncryptedDataKeys;
    _headerState;
    constructor(signaturePolicy, { commitmentPolicy, maxEncryptedDataKeys }, cmm) {
        super();
        /* Precondition: ParseHeaderStream needs a valid commitmentPolicy. */
        needs(CommitmentPolicy[commitmentPolicy], 'Invalid commitment policy.');
        /* Precondition: ParseHeaderStream needs a valid signaturePolicy. */
        needs(SignaturePolicy[signaturePolicy], 'Invalid signature policy.');
        // buildDecrypt defaults this to false for backwards compatibility, so this is satisfied
        /* Precondition: ParseHeaderStream needs a valid maxEncryptedDataKeys. */
        needs(maxEncryptedDataKeys === false || maxEncryptedDataKeys >= 1, 'Invalid maxEncryptedDataKeys value.');
        Object.defineProperty(this, 'materialsManager', {
            value: cmm,
            enumerable: true,
        });
        Object.defineProperty(this, 'commitmentPolicy', {
            value: commitmentPolicy,
            enumerable: true,
        });
        Object.defineProperty(this, 'maxEncryptedDataKeys', {
            value: maxEncryptedDataKeys,
            enumerable: true,
        });
        this._headerState = {
            buffer: Buffer.alloc(0),
            headerParsed: false,
        };
        Object.defineProperty(this, 'signaturePolicy', {
            value: signaturePolicy,
            enumerable: true,
        });
    }
    _transform(
    // eslint-disable-next-line @typescript-eslint/explicit-module-boundary-types
    chunk, encoding, callback) {
        try {
            const { _headerState, commitmentPolicy, materialsManager, signaturePolicy, maxEncryptedDataKeys, } = this;
            const { buffer } = _headerState;
            const headerBuffer = Buffer.concat([buffer, chunk]);
            const headerInfo = deserialize.deserializeMessageHeader(headerBuffer, {
                maxEncryptedDataKeys,
            });
            if (!headerInfo) {
                _headerState.buffer = headerBuffer;
                return callback();
            }
            const { messageHeader, algorithmSuite } = headerInfo;
            const messageIDStr = Buffer.from(messageHeader.messageId).toString('hex');
            /* Precondition: The parsed header algorithmSuite from ParseHeaderStream must be supported by the commitmentPolicy. */
            CommitmentPolicySuites.isDecryptEnabled(commitmentPolicy, algorithmSuite, messageIDStr);
            /* Precondition: The parsed header algorithmSuite from ParseHeaderStream must be supported by the signaturePolicy. */
            SignaturePolicySuites.isDecryptEnabled(signaturePolicy, algorithmSuite, messageIDStr);
            const { rawHeader, headerAuth } = headerInfo;
            const { headerIv, headerAuthTag, headerAuthLength } = headerAuth;
            const suite = new NodeAlgorithmSuite(algorithmSuite.id);
            const { messageId, encryptionContext, encryptedDataKeys } = messageHeader;
            materialsManager
                .decryptMaterials({ suite, encryptionContext, encryptedDataKeys })
                .then((material) => {
                /* Precondition: The material algorithmSuite returned to ParseHeaderStream must be supported by the commitmentPolicy. */
                CommitmentPolicySuites.isDecryptEnabled(commitmentPolicy, material.suite, messageIDStr);
                /* Precondition: The material algorithmSuite returned to ParseHeaderStream must be supported by the signaturePolicy. */
                SignaturePolicySuites.isDecryptEnabled(signaturePolicy, material.suite, messageIDStr);
                _headerState.buffer = Buffer.alloc(0); // clear the Buffer...
                const { getDecipherInfo, getVerify, dispose } = getDecryptionHelper(material);
                const getDecipher = getDecipherInfo(messageId, 
                /* This is sub-optimal.
                 * Ideally I could pluck the `suiteData`
                 * right off the header
                 * and in such a way that may be undefined.
                 * But that has other consequences
                 * that are beyond the scope of this course.
                 */
                messageHeader.suiteData);
                const headerAuth = getDecipher(headerIv);
                headerAuth.setAAD(Buffer.from(rawHeader.buffer, rawHeader.byteOffset, rawHeader.byteLength));
                headerAuth.setAuthTag(Buffer.from(headerAuthTag.buffer, headerAuthTag.byteOffset, headerAuthTag.byteLength));
                headerAuth.update(Buffer.alloc(0));
                headerAuth.final(); // will throw if invalid
                const verify = getVerify ? getVerify() : void 0;
                const verifyInfo = {
                    headerInfo,
                    getDecipher,
                    verify,
                    dispose,
                };
                this.emit('VerifyInfo', verifyInfo);
                this.emit('MessageHeader', headerInfo.messageHeader);
                _headerState.headerParsed = true;
                // The header is parsed, pass control
                const readPos = rawHeader.byteLength + headerAuthLength;
                const tail = headerBuffer.slice(readPos);
                /* needs calls in downstream _transform streams will throw.
                 * But streams are async.
                 * So this error should be turned into an `.emit('error', ex)`.
                 */
                this._transform = (chunk, _enc, cb) => {
                    try {
                        cb(null, chunk);
                    }
                    catch (ex) {
                        this.emit('error', ex);
                    }
                };
                // flush the tail.  Stream control is now in the verify and decrypt streams
                return setImmediate(() => this._transform(tail, encoding, callback));
            })
                .catch((err) => callback(err));
        }
        catch (ex) {
            /* Exceptional Postcondition: An error MUST be emitted or this would be an unhandled exception. */
            this.emit('error', ex);
        }
    }
    _flush(callback) {
        /* Postcondition: A completed header MUST have been processed.
         * callback is an errBack function,
         * so it expects either an error OR undefined
         */
        callback(this._headerState.headerParsed
            ? undefined
            : new Error('Incomplete Header'));
    }
}
//# sourceMappingURL=data:application/json;base64,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