// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import { serializeFactory, uInt16BE } from '@aws-crypto/serialize';
import { compare } from './portable_compare';
//  512 bits of 0 for padding between hashes in decryption materials cache ID generation.
const BIT_PAD_512 = Buffer.alloc(64);
export function buildCryptographicMaterialsCacheKeyHelpers(fromUtf8, toUtf8, sha512) {
    const { serializeEncryptionContext, serializeEncryptedDataKey } = serializeFactory(fromUtf8);
    return {
        buildEncryptionMaterialCacheKey,
        buildDecryptionMaterialCacheKey,
        encryptedDataKeysHash,
        encryptionContextHash,
    };
    async function buildEncryptionMaterialCacheKey(partition, { suite, encryptionContext }) {
        const algorithmInfo = suite
            ? [new Uint8Array([1]), uInt16BE(suite.id)]
            : [new Uint8Array([0])];
        const key = await sha512(await sha512(fromUtf8(partition)), ...algorithmInfo, await encryptionContextHash(encryptionContext));
        return toUtf8(key);
    }
    async function buildDecryptionMaterialCacheKey(partition, { suite, encryptedDataKeys, encryptionContext }) {
        const { id } = suite;
        const key = await sha512(await sha512(fromUtf8(partition)), uInt16BE(id), ...(await encryptedDataKeysHash(encryptedDataKeys)), BIT_PAD_512, await encryptionContextHash(encryptionContext));
        return toUtf8(key);
    }
    async function encryptedDataKeysHash(encryptedDataKeys) {
        const hashes = await Promise.all(encryptedDataKeys
            .map(serializeEncryptedDataKey)
            .map(async (edk) => sha512(edk)));
        return hashes.sort(compare);
    }
    async function encryptionContextHash(context) {
        /* The AAD section is uInt16BE(length) + AAD
         * see: https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/message-format.html#header-aad
         * However, the RAW Keyring wants _only_ the ADD.
         * So, I just slice off the length.
         */
        const serializedContext = serializeEncryptionContext(context).slice(2);
        return sha512(serializedContext);
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYnVpbGRfY3J5cHRvZ3JhcGhpY19tYXRlcmlhbHNfY2FjaGVfa2V5X2hlbHBlcnMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMvYnVpbGRfY3J5cHRvZ3JhcGhpY19tYXRlcmlhbHNfY2FjaGVfa2V5X2hlbHBlcnMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQUEsb0VBQW9FO0FBQ3BFLHNDQUFzQztBQVN0QyxPQUFPLEVBQUUsZ0JBQWdCLEVBQUUsUUFBUSxFQUFFLE1BQU0sdUJBQXVCLENBQUE7QUFDbEUsT0FBTyxFQUFFLE9BQU8sRUFBRSxNQUFNLG9CQUFvQixDQUFBO0FBRTVDLHlGQUF5RjtBQUN6RixNQUFNLFdBQVcsR0FBRyxNQUFNLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxDQUFBO0FBRXBDLE1BQU0sVUFBVSwwQ0FBMEMsQ0FHeEQsUUFBdUMsRUFDdkMsTUFBcUMsRUFDckMsTUFBaUU7SUFFakUsTUFBTSxFQUFFLDBCQUEwQixFQUFFLHlCQUF5QixFQUFFLEdBQzdELGdCQUFnQixDQUFDLFFBQVEsQ0FBQyxDQUFBO0lBRTVCLE9BQU87UUFDTCwrQkFBK0I7UUFDL0IsK0JBQStCO1FBQy9CLHFCQUFxQjtRQUNyQixxQkFBcUI7S0FDdEIsQ0FBQTtJQUVELEtBQUssVUFBVSwrQkFBK0IsQ0FDNUMsU0FBaUIsRUFDakIsRUFBRSxLQUFLLEVBQUUsaUJBQWlCLEVBQXdCO1FBRWxELE1BQU0sYUFBYSxHQUFHLEtBQUs7WUFDekIsQ0FBQyxDQUFDLENBQUMsSUFBSSxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLFFBQVEsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLENBQUM7WUFDM0MsQ0FBQyxDQUFDLENBQUMsSUFBSSxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUE7UUFFekIsTUFBTSxHQUFHLEdBQUcsTUFBTSxNQUFNLENBQ3RCLE1BQU0sTUFBTSxDQUFDLFFBQVEsQ0FBQyxTQUFTLENBQUMsQ0FBQyxFQUNqQyxHQUFHLGFBQWEsRUFDaEIsTUFBTSxxQkFBcUIsQ0FBQyxpQkFBaUIsQ0FBQyxDQUMvQyxDQUFBO1FBQ0QsT0FBTyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUE7SUFDcEIsQ0FBQztJQUVELEtBQUssVUFBVSwrQkFBK0IsQ0FDNUMsU0FBaUIsRUFDakIsRUFBRSxLQUFLLEVBQUUsaUJBQWlCLEVBQUUsaUJBQWlCLEVBQXdCO1FBRXJFLE1BQU0sRUFBRSxFQUFFLEVBQUUsR0FBRyxLQUFLLENBQUE7UUFFcEIsTUFBTSxHQUFHLEdBQUcsTUFBTSxNQUFNLENBQ3RCLE1BQU0sTUFBTSxDQUFDLFFBQVEsQ0FBQyxTQUFTLENBQUMsQ0FBQyxFQUNqQyxRQUFRLENBQUMsRUFBRSxDQUFDLEVBQ1osR0FBRyxDQUFDLE1BQU0scUJBQXFCLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxFQUNuRCxXQUFXLEVBQ1gsTUFBTSxxQkFBcUIsQ0FBQyxpQkFBaUIsQ0FBQyxDQUMvQyxDQUFBO1FBQ0QsT0FBTyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUE7SUFDcEIsQ0FBQztJQUVELEtBQUssVUFBVSxxQkFBcUIsQ0FDbEMsaUJBQWtEO1FBRWxELE1BQU0sTUFBTSxHQUFHLE1BQU0sT0FBTyxDQUFDLEdBQUcsQ0FDOUIsaUJBQWlCO2FBQ2QsR0FBRyxDQUFDLHlCQUF5QixDQUFDO2FBQzlCLEdBQUcsQ0FBQyxLQUFLLEVBQUUsR0FBRyxFQUFFLEVBQUUsQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FDbkMsQ0FBQTtRQUNELE9BQU8sTUFBTSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQTtJQUM3QixDQUFDO0lBRUQsS0FBSyxVQUFVLHFCQUFxQixDQUFDLE9BQTBCO1FBQzdEOzs7O1dBSUc7UUFDSCxNQUFNLGlCQUFpQixHQUFHLDBCQUEwQixDQUFDLE9BQU8sQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQTtRQUN0RSxPQUFPLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQyxDQUFBO0lBQ2xDLENBQUM7QUFDSCxDQUFDIn0=