/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.streams.converter.test;

import org.apache.streams.converter.ActivityConverterProcessor;
import org.apache.streams.core.StreamsDatum;
import org.apache.streams.jackson.StreamsJacksonMapper;
import org.apache.streams.pojo.json.Activity;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.junit.Before;
import org.junit.Test;

import java.io.IOException;
import java.util.List;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

/**
 * Test for
 * {@link org.apache.streams.converter.ActivityConverterProcessor}
 *
 * Test that default String & ObjectNode conversion works.
 */
public class NoConfigActivityConverterProcessorTest {

    private static final ObjectMapper mapper = StreamsJacksonMapper.getInstance();

    private static final String ACTIVITY_JSON = "{\"id\":\"id\",\"published\":\"Tue Jan 17 21:21:46 Z 2012\",\"verb\":\"post\",\"provider\":{\"id\":\"providerid\"}}";

    private ActivityConverterProcessor processor;

    @Before
    public void setup() {
        processor = new ActivityConverterProcessor();
        processor.prepare(null);
    }

    @Test
    public void testNoConfigActivitySerializerProcessorInvalid() {
        String INVALID_DOCUMENT = " 38Xs}";
        StreamsDatum datum = new StreamsDatum(INVALID_DOCUMENT);
        List<StreamsDatum> result = processor.process(datum);
        assertNotNull(result);
        assertEquals(0, result.size());
    }

    @Test
    public void testNoConfigActivityConverterProcessorString() {
        StreamsDatum datum = new StreamsDatum(ACTIVITY_JSON);
        List<StreamsDatum> result = processor.process(datum);
        assertNotNull(result);
        assertEquals(1, result.size());
        StreamsDatum resultDatum = result.get(0);
        assertNotNull(resultDatum);
        assertNotNull(resultDatum.getDocument());
        assertTrue(resultDatum.getDocument() instanceof Activity);
        assertTrue(((Activity)resultDatum.getDocument()).getVerb().equals("post"));
    }

    @Test
    public void testNoConfigActivitySerializerProcessorObject() throws IOException {
        ObjectNode OBJECT_DOCUMENT = mapper.readValue(ACTIVITY_JSON, ObjectNode.class);
        StreamsDatum datum = new StreamsDatum(OBJECT_DOCUMENT);
        List<StreamsDatum> result = processor.process(datum);
        assertNotNull(result);
        assertEquals(1, result.size());
        StreamsDatum resultDatum = result.get(0);
        assertNotNull(resultDatum);
        assertNotNull(resultDatum.getDocument());
        assertTrue(resultDatum.getDocument() instanceof Activity);
        assertTrue(((Activity)resultDatum.getDocument()).getVerb().equals("post"));
    }

}
