/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.broker.service.persistent;

import com.google.common.annotations.VisibleForTesting;
import io.netty.buffer.ByteBuf;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.bookkeeper.mledger.AsyncCallbacks;
import org.apache.bookkeeper.mledger.Entry;
import org.apache.bookkeeper.mledger.ManagedCursor;
import org.apache.bookkeeper.mledger.ManagedLedger;
import org.apache.bookkeeper.mledger.ManagedLedgerException;
import org.apache.bookkeeper.mledger.Position;
import org.apache.bookkeeper.mledger.impl.PositionImpl;
import org.apache.commons.lang3.StringUtils;
import org.apache.pulsar.broker.PulsarService;
import org.apache.pulsar.broker.service.Producer;
import org.apache.pulsar.broker.service.Topic;
import org.apache.pulsar.broker.service.persistent.PersistentTopic;
import org.apache.pulsar.common.api.proto.KeyValue;
import org.apache.pulsar.common.api.proto.MessageMetadata;
import org.apache.pulsar.common.protocol.Commands;
import org.apache.pulsar.common.protocol.Markers;
import org.apache.pulsar.common.util.collections.ConcurrentOpenHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MessageDeduplication {
    private final PulsarService pulsar;
    private final PersistentTopic topic;
    private final ManagedLedger managedLedger;
    private ManagedCursor managedCursor;
    private static final String IS_LAST_CHUNK = "isLastChunk";
    private volatile Status status;
    @VisibleForTesting
    final ConcurrentOpenHashMap<String, Long> highestSequencedPushed = ConcurrentOpenHashMap.newBuilder().expectedItems(16).concurrencyLevel(1).build();
    @VisibleForTesting
    final ConcurrentOpenHashMap<String, Long> highestSequencedPersisted = ConcurrentOpenHashMap.newBuilder().expectedItems(16).concurrencyLevel(1).build();
    private final int snapshotInterval;
    private int snapshotCounter;
    private volatile long lastSnapshotTimestamp = 0L;
    private final int maxNumberOfProducers;
    private final Map<String, Long> inactiveProducers = new ConcurrentHashMap<String, Long>();
    private final String replicatorPrefix;
    private final AtomicBoolean snapshotTaking = new AtomicBoolean(false);
    private static final Logger log = LoggerFactory.getLogger(MessageDeduplication.class);

    public MessageDeduplication(PulsarService pulsar, PersistentTopic topic, ManagedLedger managedLedger) {
        this.pulsar = pulsar;
        this.topic = topic;
        this.managedLedger = managedLedger;
        this.status = Status.Initialized;
        this.snapshotInterval = pulsar.getConfiguration().getBrokerDeduplicationEntriesInterval();
        this.maxNumberOfProducers = pulsar.getConfiguration().getBrokerDeduplicationMaxNumberOfProducers();
        this.snapshotCounter = 0;
        this.replicatorPrefix = pulsar.getConfiguration().getReplicatorPrefix();
    }

    private CompletableFuture<Void> recoverSequenceIdsMap() {
        this.managedCursor.getProperties().forEach((k, v) -> {
            this.producerRemoved((String)k);
            this.highestSequencedPushed.put(k, v);
            this.highestSequencedPersisted.put(k, v);
        });
        log.info("[{}] Replaying {} entries for deduplication", (Object)this.topic.getName(), (Object)this.managedCursor.getNumberOfEntries());
        CompletableFuture<Position> future = new CompletableFuture<Position>();
        this.replayCursor(future);
        return future.thenCompose(lastPosition -> {
            if (lastPosition != null && this.snapshotCounter >= this.snapshotInterval) {
                this.snapshotCounter = 0;
                return this.takeSnapshot((Position)lastPosition);
            }
            return CompletableFuture.completedFuture(null);
        });
    }

    private void replayCursor(final CompletableFuture<Position> future) {
        this.managedCursor.asyncReadEntries(100, new AsyncCallbacks.ReadEntriesCallback(){

            public void readEntriesComplete(List<Entry> entries, Object ctx) {
                Position lastPosition = null;
                for (Entry entry : entries) {
                    ByteBuf messageMetadataAndPayload = entry.getDataBuffer();
                    MessageMetadata md = Commands.parseMessageMetadata((ByteBuf)messageMetadataAndPayload);
                    String producerName = md.getProducerName();
                    long sequenceId = Math.max(md.getHighestSequenceId(), md.getSequenceId());
                    MessageDeduplication.this.highestSequencedPushed.put((Object)producerName, (Object)sequenceId);
                    MessageDeduplication.this.highestSequencedPersisted.put((Object)producerName, (Object)sequenceId);
                    MessageDeduplication.this.producerRemoved(producerName);
                    ++MessageDeduplication.this.snapshotCounter;
                    lastPosition = entry.getPosition();
                    entry.release();
                }
                if (MessageDeduplication.this.managedCursor.hasMoreEntries()) {
                    MessageDeduplication.this.pulsar.getExecutor().execute(() -> MessageDeduplication.this.replayCursor(future));
                } else {
                    future.complete(lastPosition);
                }
            }

            public void readEntriesFailed(ManagedLedgerException exception, Object ctx) {
                future.completeExceptionally(exception);
            }
        }, null, PositionImpl.LATEST);
    }

    public Status getStatus() {
        return this.status;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public CompletableFuture<Void> checkStatus() {
        boolean shouldBeEnabled = this.topic.isDeduplicationEnabled();
        MessageDeduplication messageDeduplication = this;
        synchronized (messageDeduplication) {
            if (this.status == Status.Recovering || this.status == Status.Removing) {
                this.pulsar.getExecutor().schedule(this::checkStatus, 1L, TimeUnit.MINUTES);
                return CompletableFuture.completedFuture(null);
            }
            if (this.status == Status.Initialized && !shouldBeEnabled) {
                this.status = Status.Removing;
                this.managedLedger.asyncDeleteCursor("pulsar.dedup", new AsyncCallbacks.DeleteCursorCallback(){

                    public void deleteCursorComplete(Object ctx) {
                        MessageDeduplication.this.status = Status.Disabled;
                        log.info("[{}] Deleted deduplication cursor", (Object)MessageDeduplication.this.topic.getName());
                    }

                    public void deleteCursorFailed(ManagedLedgerException exception, Object ctx) {
                        if (exception instanceof ManagedLedgerException.CursorNotFoundException) {
                            MessageDeduplication.this.status = Status.Disabled;
                        } else {
                            log.error("[{}] Deleted deduplication cursor error", (Object)MessageDeduplication.this.topic.getName(), (Object)exception);
                        }
                    }
                }, null);
            }
            if (this.status == Status.Enabled && !shouldBeEnabled) {
                final CompletableFuture<Void> future = new CompletableFuture<Void>();
                this.status = Status.Removing;
                this.managedLedger.asyncDeleteCursor("pulsar.dedup", new AsyncCallbacks.DeleteCursorCallback(){

                    public void deleteCursorComplete(Object ctx) {
                        MessageDeduplication.this.status = Status.Disabled;
                        MessageDeduplication.this.managedCursor = null;
                        MessageDeduplication.this.highestSequencedPushed.clear();
                        MessageDeduplication.this.highestSequencedPersisted.clear();
                        future.complete(null);
                        log.info("[{}] Disabled deduplication", (Object)MessageDeduplication.this.topic.getName());
                    }

                    public void deleteCursorFailed(ManagedLedgerException exception, Object ctx) {
                        if (exception instanceof ManagedLedgerException.CursorNotFoundException) {
                            MessageDeduplication.this.status = Status.Disabled;
                            MessageDeduplication.this.managedCursor = null;
                            MessageDeduplication.this.highestSequencedPushed.clear();
                            MessageDeduplication.this.highestSequencedPersisted.clear();
                            future.complete(null);
                        } else {
                            log.warn("[{}] Failed to disable deduplication: {}", (Object)MessageDeduplication.this.topic.getName(), (Object)exception.getMessage());
                            MessageDeduplication.this.status = Status.Failed;
                            future.completeExceptionally(exception);
                        }
                    }
                }, null);
                return future;
            }
            if ((this.status == Status.Disabled || this.status == Status.Initialized) && shouldBeEnabled) {
                final CompletableFuture<Void> future = new CompletableFuture<Void>();
                this.managedLedger.asyncOpenCursor("pulsar.dedup", new AsyncCallbacks.OpenCursorCallback(){

                    public void openCursorComplete(ManagedCursor cursor, Object ctx) {
                        cursor.setAlwaysInactive();
                        MessageDeduplication.this.managedCursor = cursor;
                        ((CompletableFuture)MessageDeduplication.this.recoverSequenceIdsMap().thenRun(() -> {
                            MessageDeduplication.this.status = Status.Enabled;
                            future.complete(null);
                            log.info("[{}] Enabled deduplication", (Object)MessageDeduplication.this.topic.getName());
                        })).exceptionally(ex -> {
                            MessageDeduplication.this.status = Status.Failed;
                            log.warn("[{}] Failed to enable deduplication: {}", (Object)MessageDeduplication.this.topic.getName(), (Object)ex.getMessage());
                            future.completeExceptionally((Throwable)ex);
                            return null;
                        });
                    }

                    public void openCursorFailed(ManagedLedgerException exception, Object ctx) {
                        log.warn("[{}] Failed to enable deduplication: {}", (Object)MessageDeduplication.this.topic.getName(), (Object)exception.getMessage());
                        future.completeExceptionally(exception);
                    }
                }, null);
                return future;
            }
            return CompletableFuture.completedFuture(null);
        }
    }

    public boolean isEnabled() {
        return this.status == Status.Enabled;
    }

    public MessageDupStatus isDuplicate(Topic.PublishContext publishContext, ByteBuf headersAndPayload) {
        this.setContextPropsIfRepl(publishContext, headersAndPayload);
        if (!this.isEnabled() || publishContext.isMarkerMessage()) {
            return MessageDupStatus.NotDup;
        }
        if (Producer.isRemoteOrShadow(publishContext.getProducerName(), this.replicatorPrefix)) {
            if (!publishContext.supportsReplDedupByLidAndEid()) {
                return this.isDuplicateReplV1(publishContext, headersAndPayload);
            }
            return this.isDuplicateReplV2(publishContext, headersAndPayload);
        }
        return this.isDuplicateNormal(publishContext, headersAndPayload, false);
    }

    public MessageDupStatus isDuplicateReplV1(Topic.PublishContext publishContext, ByteBuf headersAndPayload) {
        int readerIndex = headersAndPayload.readerIndex();
        MessageMetadata md = Commands.parseMessageMetadata((ByteBuf)headersAndPayload);
        headersAndPayload.readerIndex(readerIndex);
        String producerName = md.getProducerName();
        long sequenceId = md.getSequenceId();
        long highestSequenceId = Math.max(md.getHighestSequenceId(), sequenceId);
        publishContext.setOriginalProducerName(producerName);
        publishContext.setOriginalSequenceId(sequenceId);
        publishContext.setOriginalHighestSequenceId(highestSequenceId);
        return this.isDuplicateNormal(publishContext, headersAndPayload, true);
    }

    private void setContextPropsIfRepl(Topic.PublishContext publishContext, ByteBuf headersAndPayload) {
        if (publishContext.isMarkerMessage()) {
            MessageMetadata md = Commands.peekMessageMetadata((ByteBuf)headersAndPayload, (String)"Check-Deduplicate", (long)-1L);
            if (Markers.isReplicationMarker((int)md.getMarkerType())) {
                publishContext.setProperty("__MSG_PROP_IS_REPL_MARKER", "");
            }
            return;
        }
        if (Producer.isRemoteOrShadow(publishContext.getProducerName(), this.replicatorPrefix)) {
            int readerIndex = headersAndPayload.readerIndex();
            MessageMetadata md = Commands.parseMessageMetadata((ByteBuf)headersAndPayload);
            headersAndPayload.readerIndex(readerIndex);
            List kvPairList = md.getPropertiesList();
            for (KeyValue kvPair : kvPairList) {
                if (!kvPair.getKey().equals("__MSG_PROP_REPL_SOURCE_POSITION")) continue;
                if (!kvPair.getValue().contains(":")) {
                    log.warn("[{}] Unexpected {}: {}", new Object[]{publishContext.getProducerName(), "__MSG_PROP_REPL_SOURCE_POSITION", kvPair.getValue()});
                    break;
                }
                String[] ledgerIdAndEntryId = kvPair.getValue().split(":");
                if (ledgerIdAndEntryId.length != 2 || !StringUtils.isNumeric((CharSequence)ledgerIdAndEntryId[0]) || !StringUtils.isNumeric((CharSequence)ledgerIdAndEntryId[1])) {
                    log.warn("[{}] Unexpected {}: {}", new Object[]{publishContext.getProducerName(), "__MSG_PROP_REPL_SOURCE_POSITION", kvPair.getValue()});
                    break;
                }
                long[] positionPair = new long[]{Long.valueOf(ledgerIdAndEntryId[0]), Long.valueOf(ledgerIdAndEntryId[1])};
                publishContext.setProperty("__MSG_PROP_REPL_SOURCE_POSITION", positionPair);
                break;
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public MessageDupStatus isDuplicateReplV2(Topic.PublishContext publishContext, ByteBuf headersAndPayload) {
        Object positionPairObj = publishContext.getProperty("__MSG_PROP_REPL_SOURCE_POSITION");
        if (positionPairObj == null || !(positionPairObj instanceof long[])) {
            log.error("[{}] Message can not determine whether the message is duplicated due to the acquired messages props were are invalid. producer={}. supportsReplDedupByLidAndEid: {}, sequence-id {}, prop-{}: not in expected format", new Object[]{this.topic.getName(), publishContext.getProducerName(), publishContext.supportsReplDedupByLidAndEid(), publishContext.getSequenceId(), "__MSG_PROP_REPL_SOURCE_POSITION"});
            return MessageDupStatus.Unknown;
        }
        long[] positionPair = (long[])positionPairObj;
        long replSequenceLId = positionPair[0];
        long replSequenceEId = positionPair[1];
        String lastSequenceLIdKey = publishContext.getProducerName() + "_LID";
        String lastSequenceEIdKey = publishContext.getProducerName() + "_EID";
        ConcurrentOpenHashMap<String, Long> concurrentOpenHashMap = this.highestSequencedPushed;
        synchronized (concurrentOpenHashMap) {
            Long lastSequenceLIdPushed = (Long)this.highestSequencedPushed.get((Object)lastSequenceLIdKey);
            Long lastSequenceEIdPushed = (Long)this.highestSequencedPushed.get((Object)lastSequenceEIdKey);
            if (lastSequenceLIdPushed != null && lastSequenceEIdPushed != null && (replSequenceLId < lastSequenceLIdPushed || replSequenceLId == lastSequenceLIdPushed && replSequenceEId <= lastSequenceEIdPushed)) {
                if (log.isDebugEnabled()) {
                    log.debug("[{}] Message identified as duplicated producer={}. publishing {}:{}, latest publishing in-progress {}:{}", new Object[]{this.topic.getName(), publishContext.getProducerName(), lastSequenceLIdPushed, lastSequenceEIdPushed, lastSequenceLIdPushed, lastSequenceEIdPushed});
                }
                Long lastSequenceLIdPersisted = (Long)this.highestSequencedPersisted.get((Object)lastSequenceLIdKey);
                Long lastSequenceEIdPersisted = (Long)this.highestSequencedPersisted.get((Object)lastSequenceEIdKey);
                if (log.isDebugEnabled()) {
                    log.debug("[{}] Message identified as duplicated producer={}. publishing {}:{}, latest persisted {}:{}", new Object[]{this.topic.getName(), publishContext.getProducerName(), replSequenceLId, replSequenceEId, lastSequenceLIdPersisted, lastSequenceEIdPersisted});
                }
                if (lastSequenceLIdPersisted != null && lastSequenceEIdPersisted != null && (replSequenceLId < lastSequenceLIdPersisted || replSequenceLId == lastSequenceLIdPersisted && replSequenceEId <= lastSequenceEIdPersisted)) {
                    return MessageDupStatus.Dup;
                }
                return MessageDupStatus.Unknown;
            }
            this.highestSequencedPushed.put((Object)lastSequenceLIdKey, (Object)replSequenceLId);
            this.highestSequencedPushed.put((Object)lastSequenceEIdKey, (Object)replSequenceEId);
        }
        if (log.isDebugEnabled()) {
            log.debug("[{}] Message identified as non-duplicated producer={}. publishing {}:{}", new Object[]{this.topic.getName(), publishContext.getProducerName(), replSequenceLId, replSequenceEId});
        }
        return MessageDupStatus.NotDup;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public MessageDupStatus isDuplicateNormal(Topic.PublishContext publishContext, ByteBuf headersAndPayload, boolean useOriginalProducerName) {
        String producerName = publishContext.getProducerName();
        if (useOriginalProducerName) {
            producerName = publishContext.getOriginalProducerName();
        }
        long sequenceId = publishContext.getSequenceId();
        long highestSequenceId = Math.max(publishContext.getHighestSequenceId(), sequenceId);
        MessageMetadata md = null;
        long chunkID = -1L;
        long totalChunk = -1L;
        if (publishContext.isChunked()) {
            if (md == null) {
                int readerIndex = headersAndPayload.readerIndex();
                md = Commands.parseMessageMetadata((ByteBuf)headersAndPayload);
                headersAndPayload.readerIndex(readerIndex);
            }
            chunkID = md.getChunkId();
            totalChunk = md.getNumChunksFromMsg();
        }
        if (chunkID != -1L && chunkID != totalChunk - 1L) {
            publishContext.setProperty(IS_LAST_CHUNK, Boolean.FALSE);
            return MessageDupStatus.NotDup;
        }
        ConcurrentOpenHashMap<String, Long> concurrentOpenHashMap = this.highestSequencedPushed;
        synchronized (concurrentOpenHashMap) {
            Long lastSequenceIdPushed = (Long)this.highestSequencedPushed.get((Object)producerName);
            if (lastSequenceIdPushed != null && sequenceId <= lastSequenceIdPushed) {
                Long lastSequenceIdPersisted;
                if (log.isDebugEnabled()) {
                    log.debug("[{}] Message identified as duplicated producer={} seq-id={} -- highest-seq-id={}", new Object[]{this.topic.getName(), producerName, sequenceId, lastSequenceIdPushed});
                }
                if ((lastSequenceIdPersisted = (Long)this.highestSequencedPersisted.get((Object)producerName)) != null && sequenceId <= lastSequenceIdPersisted) {
                    return MessageDupStatus.Dup;
                }
                return MessageDupStatus.Unknown;
            }
            this.highestSequencedPushed.put((Object)producerName, (Object)highestSequenceId);
        }
        if (chunkID != -1L && chunkID == totalChunk - 1L) {
            publishContext.setProperty(IS_LAST_CHUNK, Boolean.TRUE);
        }
        return MessageDupStatus.NotDup;
    }

    public void recordMessagePersisted(Topic.PublishContext publishContext, PositionImpl position) {
        if (!this.isEnabled() || publishContext.isMarkerMessage()) {
            return;
        }
        if (publishContext.getProducerName().startsWith(this.replicatorPrefix) && publishContext.supportsReplDedupByLidAndEid()) {
            this.recordMessagePersistedRepl(publishContext, (Position)position);
        } else {
            this.recordMessagePersistedNormal(publishContext, (Position)position);
        }
    }

    public void recordMessagePersistedRepl(Topic.PublishContext publishContext, Position position) {
        Object positionPairObj = publishContext.getProperty("__MSG_PROP_REPL_SOURCE_POSITION");
        if (positionPairObj == null || !(positionPairObj instanceof long[])) {
            log.error("[{}] Can not persist highest sequence-id due to the acquired messages props are invalid. producer={}. supportsReplDedupByLidAndEid: {}, sequence-id {}, prop-{}: not in expected format", new Object[]{this.topic.getName(), publishContext.getProducerName(), publishContext.supportsReplDedupByLidAndEid(), publishContext.getSequenceId(), "__MSG_PROP_REPL_SOURCE_POSITION"});
            this.recordMessagePersistedNormal(publishContext, position);
            return;
        }
        long[] positionPair = (long[])positionPairObj;
        long replSequenceLId = positionPair[0];
        long replSequenceEId = positionPair[1];
        String lastSequenceLIdKey = publishContext.getProducerName() + "_LID";
        String lastSequenceEIdKey = publishContext.getProducerName() + "_EID";
        this.highestSequencedPersisted.put((Object)lastSequenceLIdKey, (Object)replSequenceLId);
        this.highestSequencedPersisted.put((Object)lastSequenceEIdKey, (Object)replSequenceEId);
        this.increaseSnapshotCounterAndTakeSnapshotIfNeeded(position);
    }

    public void recordMessagePersistedNormal(Topic.PublishContext publishContext, Position position) {
        Boolean isLastChunk;
        String producerName = publishContext.getProducerName();
        long sequenceId = publishContext.getSequenceId();
        long highestSequenceId = publishContext.getHighestSequenceId();
        if (publishContext.getOriginalProducerName() != null) {
            producerName = publishContext.getOriginalProducerName();
            sequenceId = publishContext.getOriginalSequenceId();
            highestSequenceId = publishContext.getOriginalHighestSequenceId();
        }
        if ((isLastChunk = (Boolean)publishContext.getProperty(IS_LAST_CHUNK)) == null || isLastChunk.booleanValue()) {
            this.highestSequencedPersisted.put((Object)producerName, (Object)Math.max(highestSequenceId, sequenceId));
        }
        this.increaseSnapshotCounterAndTakeSnapshotIfNeeded(position);
    }

    private void increaseSnapshotCounterAndTakeSnapshotIfNeeded(Position position) {
        if (++this.snapshotCounter >= this.snapshotInterval) {
            this.snapshotCounter = 0;
            this.takeSnapshot(position);
        } else if (log.isDebugEnabled()) {
            log.debug("[{}] Waiting for sequence-id snapshot {}/{}", new Object[]{this.topic.getName(), this.snapshotCounter, this.snapshotInterval});
        }
    }

    public void resetHighestSequenceIdPushed() {
        if (!this.isEnabled()) {
            return;
        }
        this.highestSequencedPushed.clear();
        for (String producer : this.highestSequencedPersisted.keys()) {
            this.highestSequencedPushed.put((Object)producer, (Object)((Long)this.highestSequencedPersisted.get((Object)producer)));
        }
    }

    private CompletableFuture<Void> takeSnapshot(final Position position) {
        final CompletableFuture<Void> future = new CompletableFuture<Void>();
        if (log.isDebugEnabled()) {
            log.debug("[{}] Taking snapshot of sequence ids map", (Object)this.topic.getName());
        }
        if (!this.snapshotTaking.compareAndSet(false, true)) {
            future.complete(null);
            return future;
        }
        TreeMap snapshot = new TreeMap();
        this.highestSequencedPersisted.forEach((producerName, sequenceId) -> {
            if (snapshot.size() < this.maxNumberOfProducers) {
                snapshot.put(producerName, sequenceId);
            }
        });
        this.getManagedCursor().asyncMarkDelete(position, snapshot, new AsyncCallbacks.MarkDeleteCallback(){

            public void markDeleteComplete(Object ctx) {
                if (log.isDebugEnabled()) {
                    log.debug("[{}] Stored new deduplication snapshot at {}", (Object)MessageDeduplication.this.topic.getName(), (Object)position);
                }
                MessageDeduplication.this.lastSnapshotTimestamp = System.currentTimeMillis();
                MessageDeduplication.this.snapshotTaking.set(false);
                future.complete(null);
            }

            public void markDeleteFailed(ManagedLedgerException exception, Object ctx) {
                log.warn("[{}] Failed to store new deduplication snapshot at {}", (Object)MessageDeduplication.this.topic.getName(), (Object)position);
                MessageDeduplication.this.snapshotTaking.set(false);
                future.completeExceptionally(exception);
            }
        }, null);
        return future;
    }

    public void producerAdded(String producerName) {
        if (!this.isEnabled()) {
            return;
        }
        this.inactiveProducers.remove(producerName);
    }

    public void producerRemoved(String producerName) {
        if (!this.isEnabled()) {
            return;
        }
        this.inactiveProducers.put(producerName, System.currentTimeMillis());
    }

    public synchronized void purgeInactiveProducers() {
        long minimumActiveTimestamp = System.currentTimeMillis() - TimeUnit.MINUTES.toMillis(this.pulsar.getConfiguration().getBrokerDeduplicationProducerInactivityTimeoutMinutes());
        if (!this.isEnabled()) {
            if (!this.inactiveProducers.isEmpty()) {
                this.inactiveProducers.clear();
            }
            return;
        }
        Iterator<Map.Entry<String, Long>> mapIterator = this.inactiveProducers.entrySet().iterator();
        boolean hasInactive = false;
        while (mapIterator.hasNext()) {
            Map.Entry<String, Long> entry = mapIterator.next();
            String producerName = entry.getKey();
            long lastActiveTimestamp = entry.getValue();
            if (lastActiveTimestamp >= minimumActiveTimestamp) continue;
            log.info("[{}] Purging dedup information for producer {}", (Object)this.topic.getName(), (Object)producerName);
            mapIterator.remove();
            this.highestSequencedPushed.remove((Object)producerName);
            this.highestSequencedPersisted.remove((Object)producerName);
            hasInactive = true;
        }
        if (hasInactive && this.isEnabled()) {
            this.takeSnapshot(this.getManagedCursor().getMarkDeletedPosition());
        }
    }

    public long getLastPublishedSequenceId(String producerName) {
        Long sequenceId = (Long)this.highestSequencedPushed.get((Object)producerName);
        return sequenceId != null ? sequenceId : -1L;
    }

    public void takeSnapshot() {
        if (!this.isEnabled()) {
            return;
        }
        Integer interval = (Integer)this.topic.getHierarchyTopicPolicies().getDeduplicationSnapshotIntervalSeconds().get();
        long currentTimeStamp = System.currentTimeMillis();
        if (interval == null || interval <= 0 || currentTimeStamp - this.lastSnapshotTimestamp < TimeUnit.SECONDS.toMillis(interval.intValue())) {
            return;
        }
        PositionImpl position = (PositionImpl)this.managedLedger.getLastConfirmedEntry();
        if (position == null) {
            return;
        }
        PositionImpl markDeletedPosition = (PositionImpl)this.managedCursor.getMarkDeletedPosition();
        if (markDeletedPosition != null && position.compareTo(markDeletedPosition) <= 0) {
            return;
        }
        this.takeSnapshot((Position)position);
    }

    @VisibleForTesting
    ManagedCursor getManagedCursor() {
        return this.managedCursor;
    }

    @VisibleForTesting
    Map<String, Long> getInactiveProducers() {
        return this.inactiveProducers;
    }

    static enum Status {
        Initialized,
        Disabled,
        Recovering,
        Enabled,
        Removing,
        Failed;

    }

    @VisibleForTesting
    public static enum MessageDupStatus {
        Unknown,
        NotDup,
        Dup;

    }

    public static class MessageDupUnknownException
    extends RuntimeException {
        public MessageDupUnknownException() {
            super("Cannot determine whether the message is a duplicate at this time");
        }
    }
}

