/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.server.raftlog.segmented;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentNavigableMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import org.apache.ratis.proto.RaftProtos;
import org.apache.ratis.server.RaftServerConfigKeys;
import org.apache.ratis.server.metrics.SegmentedRaftLogMetrics;
import org.apache.ratis.server.protocol.TermIndex;
import org.apache.ratis.server.raftlog.LogEntryHeader;
import org.apache.ratis.server.raftlog.LogProtoUtils;
import org.apache.ratis.server.raftlog.RaftLogIOException;
import org.apache.ratis.server.raftlog.segmented.LogSegmentStartEnd;
import org.apache.ratis.server.raftlog.segmented.SegmentedRaftLogFormat;
import org.apache.ratis.server.raftlog.segmented.SegmentedRaftLogInputStream;
import org.apache.ratis.server.storage.RaftStorage;
import org.apache.ratis.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.apache.ratis.thirdparty.com.google.protobuf.CodedOutputStream;
import org.apache.ratis.util.FileUtils;
import org.apache.ratis.util.JavaUtils;
import org.apache.ratis.util.Preconditions;
import org.apache.ratis.util.ReferenceCountedObject;
import org.apache.ratis.util.SizeInBytes;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class LogSegment {
    static final Logger LOG = LoggerFactory.getLogger(LogSegment.class);
    private volatile boolean isOpen;
    private long totalFileSize = SegmentedRaftLogFormat.getHeaderLength();
    private final long startIndex;
    private volatile long endIndex;
    private final RaftStorage storage;
    private final SizeInBytes maxOpSize;
    private final LogEntryLoader cacheLoader;
    private final AtomicInteger loadingTimes = new AtomicInteger();
    private final Records records = new Records();
    private final EntryCache entryCache = new EntryCache();
    static final Comparator<Object> SEGMENT_TO_INDEX_COMPARATOR = (o1, o2) -> {
        if (o1 instanceof LogSegment && o2 instanceof Long) {
            return ((LogSegment)o1).compareTo((Long)o2);
        }
        if (o1 instanceof Long && o2 instanceof LogSegment) {
            return Integer.compare(0, ((LogSegment)o2).compareTo((Long)o1));
        }
        throw new IllegalStateException("Unexpected objects to compare(" + o1 + "," + o2 + ")");
    };

    static long getEntrySize(RaftProtos.LogEntryProto entry, Op op) {
        switch (op) {
            case CHECK_SEGMENT_FILE_FULL: 
            case LOAD_SEGMENT_FILE: 
            case WRITE_CACHE_WITH_STATE_MACHINE_CACHE: {
                Preconditions.assertTrue((!LogProtoUtils.hasStateMachineData(entry) ? 1 : 0) != 0, () -> "Unexpected LogEntryProto with StateMachine data: op=" + (Object)((Object)op) + ", entry=" + entry);
                break;
            }
            case WRITE_CACHE_WITHOUT_STATE_MACHINE_CACHE: 
            case REMOVE_CACHE: {
                break;
            }
            default: {
                throw new IllegalStateException("Unexpected op " + (Object)((Object)op) + ", entry=" + entry);
            }
        }
        int serialized = entry.getSerializedSize();
        return (long)(serialized + CodedOutputStream.computeUInt32SizeNoTag((int)serialized)) + 4L;
    }

    static LogSegment newOpenSegment(RaftStorage storage, long start, SizeInBytes maxOpSize, SegmentedRaftLogMetrics raftLogMetrics) {
        Preconditions.assertTrue((start >= 0L ? 1 : 0) != 0);
        return new LogSegment(storage, true, start, start - 1L, maxOpSize, raftLogMetrics);
    }

    @VisibleForTesting
    static LogSegment newCloseSegment(RaftStorage storage, long start, long end, SizeInBytes maxOpSize, SegmentedRaftLogMetrics raftLogMetrics) {
        Preconditions.assertTrue((start >= 0L && end >= start ? 1 : 0) != 0);
        return new LogSegment(storage, false, start, end, maxOpSize, raftLogMetrics);
    }

    static LogSegment newLogSegment(RaftStorage storage, LogSegmentStartEnd startEnd, SizeInBytes maxOpSize, SegmentedRaftLogMetrics metrics) {
        return startEnd.isOpen() ? LogSegment.newOpenSegment(storage, startEnd.getStartIndex(), maxOpSize, metrics) : LogSegment.newCloseSegment(storage, startEnd.getStartIndex(), startEnd.getEndIndex(), maxOpSize, metrics);
    }

    public static int readSegmentFile(File file, LogSegmentStartEnd startEnd, SizeInBytes maxOpSize, RaftServerConfigKeys.Log.CorruptionPolicy corruptionPolicy, SegmentedRaftLogMetrics raftLogMetrics, Consumer<ReferenceCountedObject<RaftProtos.LogEntryProto>> entryConsumer) throws IOException {
        int count = 0;
        try (SegmentedRaftLogInputStream in = new SegmentedRaftLogInputStream(file, startEnd, maxOpSize, raftLogMetrics);){
            RaftProtos.LogEntryProto next;
            RaftProtos.LogEntryProto prev = null;
            while ((next = in.nextEntry()) != null) {
                if (prev != null) {
                    Preconditions.assertTrue((next.getIndex() == prev.getIndex() + 1L ? 1 : 0) != 0, (String)"gap between entry %s and entry %s", (Object[])new Object[]{prev, next});
                }
                if (entryConsumer != null) {
                    entryConsumer.accept((ReferenceCountedObject<RaftProtos.LogEntryProto>)ReferenceCountedObject.wrap((Object)next));
                }
                ++count;
                prev = next;
            }
        }
        catch (IOException ioe) {
            switch (corruptionPolicy) {
                case EXCEPTION: {
                    throw ioe;
                }
                case WARN_AND_RETURN: {
                    LOG.warn("Failed to read segment file {} ({}): only {} entries read successfully", new Object[]{file, startEnd, count, ioe});
                    break;
                }
                default: {
                    throw new IllegalStateException("Unexpected enum value: " + corruptionPolicy + ", class=" + RaftServerConfigKeys.Log.CorruptionPolicy.class);
                }
            }
        }
        return count;
    }

    static LogSegment loadSegment(RaftStorage storage, File file, LogSegmentStartEnd startEnd, SizeInBytes maxOpSize, boolean keepEntryInCache, Consumer<RaftProtos.LogEntryProto> logConsumer, SegmentedRaftLogMetrics raftLogMetrics) throws IOException {
        boolean corrupted;
        LogSegment segment = LogSegment.newLogSegment(storage, startEnd, maxOpSize, raftLogMetrics);
        RaftServerConfigKeys.Log.CorruptionPolicy corruptionPolicy = RaftServerConfigKeys.Log.CorruptionPolicy.get((Object)storage, RaftStorage::getLogCorruptionPolicy);
        boolean isOpen = startEnd.isOpen();
        int entryCount = LogSegment.readSegmentFile(file, startEnd, maxOpSize, corruptionPolicy, raftLogMetrics, entry -> segment.append(Op.LOAD_SEGMENT_FILE, (ReferenceCountedObject<RaftProtos.LogEntryProto>)entry, keepEntryInCache || isOpen, logConsumer));
        LOG.info("Successfully read {} entries from segment file {}", (Object)entryCount, (Object)file);
        long start = startEnd.getStartIndex();
        long end = isOpen ? segment.getEndIndex() : startEnd.getEndIndex();
        int expectedEntryCount = Math.toIntExact(end - start + 1L);
        boolean bl = corrupted = entryCount != expectedEntryCount;
        if (corrupted) {
            LOG.warn("Segment file is corrupted: expected to have {} entries but only {} entries read successfully", (Object)expectedEntryCount, (Object)entryCount);
        }
        if (entryCount == 0) {
            Path deleted = FileUtils.deleteFile((File)file);
            LOG.info("Deleted RaftLog segment since entry count is zero: startEnd={}, path={}", (Object)startEnd, (Object)deleted);
            return null;
        }
        if (file.length() > segment.getTotalFileSize()) {
            FileUtils.truncateFile((File)file, (long)segment.getTotalFileSize());
        }
        try {
            segment.assertSegment(start, entryCount, corrupted, end);
        }
        catch (Exception e) {
            throw new IllegalStateException("Failed to read segment file " + file, e);
        }
        return segment;
    }

    private void assertSegment(long expectedStart, int expectedEntryCount, boolean corrupted, long expectedEnd) {
        Preconditions.assertSame((long)expectedStart, (long)this.getStartIndex(), (String)"Segment start index");
        Preconditions.assertSame((int)expectedEntryCount, (int)this.records.size(), (String)"Number of records");
        long expectedLastIndex = expectedStart + (long)expectedEntryCount - 1L;
        Preconditions.assertSame((long)expectedLastIndex, (long)this.getEndIndex(), (String)"Segment end index");
        LogRecord last = this.records.getLast();
        if (last != null) {
            Preconditions.assertSame((long)expectedLastIndex, (long)last.getTermIndex().getIndex(), (String)"Index at the last record");
            LogRecord first = this.records.getFirst();
            Objects.requireNonNull(first, "first record");
            Preconditions.assertSame((long)expectedStart, (long)first.getTermIndex().getIndex(), (String)"Index at the first record");
        }
        if (!corrupted) {
            Preconditions.assertSame((long)expectedEnd, (long)expectedLastIndex, (String)"End/last Index");
        }
    }

    File getFile() {
        return LogSegmentStartEnd.valueOf(this.startIndex, this.endIndex, this.isOpen).getFile(this.storage);
    }

    private LogSegment(RaftStorage storage, boolean isOpen, long start, long end, SizeInBytes maxOpSize, SegmentedRaftLogMetrics raftLogMetrics) {
        this.storage = storage;
        this.isOpen = isOpen;
        this.startIndex = start;
        this.endIndex = end;
        this.maxOpSize = maxOpSize;
        this.cacheLoader = new LogEntryLoader(raftLogMetrics);
    }

    long getStartIndex() {
        return this.startIndex;
    }

    long getEndIndex() {
        return this.endIndex;
    }

    boolean isOpen() {
        return this.isOpen;
    }

    int numOfEntries() {
        return Math.toIntExact(this.endIndex - this.startIndex + 1L);
    }

    RaftServerConfigKeys.Log.CorruptionPolicy getLogCorruptionPolicy() {
        return RaftServerConfigKeys.Log.CorruptionPolicy.get((Object)this.storage, RaftStorage::getLogCorruptionPolicy);
    }

    void appendToOpenSegment(Op op, ReferenceCountedObject<RaftProtos.LogEntryProto> entryRef) {
        Preconditions.assertTrue((boolean)this.isOpen(), (String)"The log segment %s is not open for append", (Object[])new Object[]{this});
        this.append(op, entryRef, true, null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void append(Op op, ReferenceCountedObject<RaftProtos.LogEntryProto> entryRef, boolean keepEntryInCache, Consumer<RaftProtos.LogEntryProto> logConsumer) {
        RaftProtos.LogEntryProto entry = (RaftProtos.LogEntryProto)entryRef.retain();
        try {
            LogRecord record = new LogRecord(this.totalFileSize, entry);
            if (keepEntryInCache) {
                this.putEntryCache(record.getTermIndex(), entryRef, op);
            }
            this.appendLogRecord(op, record);
            this.totalFileSize += LogSegment.getEntrySize(entry, op);
            if (logConsumer != null) {
                logConsumer.accept(entry);
            }
        }
        finally {
            entryRef.release();
        }
    }

    private void appendLogRecord(Op op, LogRecord record) {
        Objects.requireNonNull(record, "record == null");
        LogRecord currentLast = this.records.getLast();
        long index = record.getTermIndex().getIndex();
        if (currentLast == null) {
            Preconditions.assertTrue((index == this.startIndex ? 1 : 0) != 0, (String)"%s: gap between start index %s and the entry to append %s", (Object[])new Object[]{op, this.startIndex, index});
        } else {
            long currentLastIndex = currentLast.getTermIndex().getIndex();
            Preconditions.assertTrue((index == currentLastIndex + 1L ? 1 : 0) != 0, (String)"%s: gap between last entry %s and the entry to append %s", (Object[])new Object[]{op, currentLastIndex, index});
        }
        this.records.append(record);
        this.endIndex = index;
    }

    ReferenceCountedObject<RaftProtos.LogEntryProto> getEntryFromCache(TermIndex ti) {
        return this.entryCache.get(ti);
    }

    synchronized ReferenceCountedObject<RaftProtos.LogEntryProto> loadCache(TermIndex ti) throws RaftLogIOException {
        ReferenceCountedObject<RaftProtos.LogEntryProto> entry = this.entryCache.get(ti);
        if (entry != null) {
            try {
                entry.retain();
                return entry;
            }
            catch (IllegalStateException illegalStateException) {
                // empty catch block
            }
        }
        try {
            return this.cacheLoader.load(ti);
        }
        catch (RaftLogIOException e) {
            throw e;
        }
        catch (Exception e) {
            throw new RaftLogIOException("Failed to loadCache for log entry " + ti, (Throwable)e);
        }
    }

    LogRecord getLogRecord(long index) {
        return this.records.get(index);
    }

    TermIndex getLastTermIndex() {
        LogRecord last = this.records.getLast();
        return last == null ? null : last.getTermIndex();
    }

    long getTotalFileSize() {
        return this.totalFileSize;
    }

    long getTotalCacheSize() {
        return this.entryCache.size();
    }

    synchronized void truncate(long fromIndex) {
        Preconditions.assertTrue((fromIndex >= this.startIndex && fromIndex <= this.endIndex ? 1 : 0) != 0);
        for (long index = this.endIndex; index >= fromIndex; --index) {
            LogRecord removed = this.records.removeLast();
            Preconditions.assertSame((long)index, (long)removed.getTermIndex().getIndex(), (String)"removedIndex");
            this.removeEntryCache(removed.getTermIndex());
            this.totalFileSize = removed.offset;
        }
        this.isOpen = false;
        this.endIndex = fromIndex - 1L;
    }

    void close() {
        Preconditions.assertTrue((boolean)this.isOpen());
        this.isOpen = false;
    }

    public String toString() {
        return this.isOpen() ? "log_inprogress_" + this.startIndex : "log-" + this.startIndex + "_" + this.endIndex;
    }

    private int compareTo(Long l) {
        return l >= this.getStartIndex() && l <= this.getEndIndex() ? 0 : (this.getEndIndex() < l ? -1 : 1);
    }

    synchronized void clear() {
        this.records.clear();
        this.entryCache.close();
        this.endIndex = this.startIndex - 1L;
    }

    int getLoadingTimes() {
        return this.loadingTimes.get();
    }

    void evictCache() {
        this.entryCache.evict();
    }

    void putEntryCache(TermIndex key, ReferenceCountedObject<RaftProtos.LogEntryProto> valueRef, Op op) {
        this.entryCache.put(key, valueRef, op);
    }

    void removeEntryCache(TermIndex key) {
        this.entryCache.remove(key);
    }

    boolean hasCache() {
        return this.isOpen || this.entryCache.size() > 0L;
    }

    boolean containsIndex(long index) {
        return this.startIndex <= index && this.endIndex >= index;
    }

    boolean hasEntries() {
        return this.numOfEntries() > 0;
    }

    class EntryCache {
        private Map<TermIndex, Item> map = new HashMap<TermIndex, Item>();
        private final AtomicLong size = new AtomicLong();

        EntryCache() {
        }

        public String toString() {
            return JavaUtils.getClassSimpleName(this.getClass()) + "-" + LogSegment.this;
        }

        long size() {
            return this.size.get();
        }

        synchronized ReferenceCountedObject<RaftProtos.LogEntryProto> get(TermIndex ti) {
            if (this.map == null) {
                return null;
            }
            Item ref = this.map.get(ti);
            return ref == null ? null : ref.get();
        }

        synchronized void close() {
            if (this.map == null) {
                return;
            }
            this.evict();
            this.map = null;
            LOG.info("Successfully closed {}", (Object)this);
        }

        synchronized void evict() {
            if (this.map == null) {
                return;
            }
            Iterator<Map.Entry<TermIndex, Item>> i = this.map.entrySet().iterator();
            while (i.hasNext()) {
                this.release(i.next().getValue());
                i.remove();
            }
        }

        synchronized void put(TermIndex key, ReferenceCountedObject<RaftProtos.LogEntryProto> valueRef, Op op) {
            if (this.map == null) {
                return;
            }
            valueRef.retain();
            long serializedSize = LogSegment.getEntrySize((RaftProtos.LogEntryProto)valueRef.get(), op);
            this.release(this.map.put(key, new Item(valueRef, serializedSize)));
            this.size.getAndAdd(serializedSize);
        }

        private void release(Item ref) {
            if (ref == null) {
                return;
            }
            long serializedSize = ref.release();
            this.size.getAndAdd(-serializedSize);
        }

        synchronized void remove(TermIndex key) {
            if (this.map == null) {
                return;
            }
            this.release(this.map.remove(key));
        }
    }

    private static class Item {
        private final AtomicReference<ReferenceCountedObject<RaftProtos.LogEntryProto>> ref;
        private final long serializedSize;

        Item(ReferenceCountedObject<RaftProtos.LogEntryProto> obj, long serializedSize) {
            this.ref = new AtomicReference<ReferenceCountedObject<RaftProtos.LogEntryProto>>(obj);
            this.serializedSize = serializedSize;
        }

        ReferenceCountedObject<RaftProtos.LogEntryProto> get() {
            return this.ref.get();
        }

        long release() {
            ReferenceCountedObject entry = this.ref.getAndSet(null);
            if (entry == null) {
                return 0L;
            }
            entry.release();
            return this.serializedSize;
        }
    }

    class LogEntryLoader {
        private final SegmentedRaftLogMetrics raftLogMetrics;

        LogEntryLoader(SegmentedRaftLogMetrics raftLogMetrics) {
            this.raftLogMetrics = raftLogMetrics;
        }

        ReferenceCountedObject<RaftProtos.LogEntryProto> load(TermIndex key) throws IOException {
            File file = LogSegment.this.getFile();
            AtomicReference toReturn = new AtomicReference();
            LogSegmentStartEnd startEnd = LogSegmentStartEnd.valueOf(LogSegment.this.startIndex, LogSegment.this.endIndex, LogSegment.this.isOpen);
            LogSegment.readSegmentFile(file, startEnd, LogSegment.this.maxOpSize, LogSegment.this.getLogCorruptionPolicy(), this.raftLogMetrics, entryRef -> {
                RaftProtos.LogEntryProto entry = (RaftProtos.LogEntryProto)entryRef.retain();
                try {
                    TermIndex ti = TermIndex.valueOf((RaftProtos.LogEntryProto)entry);
                    LogSegment.this.putEntryCache(ti, (ReferenceCountedObject<RaftProtos.LogEntryProto>)entryRef, Op.LOAD_SEGMENT_FILE);
                    if (ti.equals(key)) {
                        entryRef.retain();
                        toReturn.set(entryRef);
                    }
                }
                finally {
                    entryRef.release();
                }
            });
            LogSegment.this.loadingTimes.incrementAndGet();
            ReferenceCountedObject proto = (ReferenceCountedObject)toReturn.get();
            if (proto == null) {
                throw new RaftLogIOException("Failed to load log entry " + key);
            }
            return proto;
        }
    }

    private static class Records {
        private final ConcurrentNavigableMap<Long, LogRecord> map = new ConcurrentSkipListMap<Long, LogRecord>();

        private Records() {
        }

        int size() {
            return this.map.size();
        }

        LogRecord getFirst() {
            Map.Entry first = this.map.firstEntry();
            return first != null ? (LogRecord)first.getValue() : null;
        }

        LogRecord getLast() {
            Map.Entry last = this.map.lastEntry();
            return last != null ? (LogRecord)last.getValue() : null;
        }

        LogRecord get(long i) {
            return (LogRecord)this.map.get(i);
        }

        long append(LogRecord record) {
            long index = record.getTermIndex().getIndex();
            LogRecord previous = this.map.put(index, record);
            Preconditions.assertNull((Object)previous, (String)"previous");
            return index;
        }

        LogRecord removeLast() {
            Map.Entry last = this.map.pollLastEntry();
            return (LogRecord)Objects.requireNonNull(last, "last == null").getValue();
        }

        void clear() {
            this.map.clear();
        }
    }

    static class LogRecord {
        private final long offset;
        private final LogEntryHeader logEntryHeader;

        LogRecord(long offset, RaftProtos.LogEntryProto entry) {
            this.offset = offset;
            this.logEntryHeader = LogEntryHeader.valueOf((RaftProtos.LogEntryProto)entry);
        }

        LogEntryHeader getLogEntryHeader() {
            return this.logEntryHeader;
        }

        TermIndex getTermIndex() {
            return this.getLogEntryHeader().getTermIndex();
        }

        long getOffset() {
            return this.offset;
        }
    }

    static enum Op {
        LOAD_SEGMENT_FILE,
        REMOVE_CACHE,
        CHECK_SEGMENT_FILE_FULL,
        WRITE_CACHE_WITH_STATE_MACHINE_CACHE,
        WRITE_CACHE_WITHOUT_STATE_MACHINE_CACHE;

    }
}

