/*
 * Decompiled with CFR 0.152.
 */
package org.apache.gravitino.lock;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.time.StopWatch;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.gravitino.Config;
import org.apache.gravitino.Configs;
import org.apache.gravitino.NameIdentifier;
import org.apache.gravitino.lock.TreeLock;
import org.apache.gravitino.lock.TreeLockNode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LockManager {
    private static final Logger LOG = LoggerFactory.getLogger(LockManager.class);
    static final NameIdentifier ROOT = NameIdentifier.of((String[])new String[]{"/"});
    @VisibleForTesting
    TreeLockNode treeLockRootNode;
    final AtomicLong totalNodeCount = new AtomicLong(1L);
    long maxTreeNodeInMemory;
    @VisibleForTesting
    long minTreeNodeInMemory;
    @VisibleForTesting
    long cleanTreeNodeIntervalInSecs;

    private void initParameters(Config config) {
        long maxNodesInMemory = config.get(Configs.TREE_LOCK_MAX_NODE_IN_MEMORY);
        if (maxNodesInMemory <= 0L) {
            throw new IllegalArgumentException(String.format("The maximum number of tree lock nodes '%d' should be greater than 0", maxNodesInMemory));
        }
        long minNodesInMemory = config.get(Configs.TREE_LOCK_MIN_NODE_IN_MEMORY);
        if (minNodesInMemory <= 0L) {
            throw new IllegalArgumentException(String.format("The minimum number of tree lock nodes '%d' should be greater than 0", minNodesInMemory));
        }
        if (maxNodesInMemory <= minNodesInMemory) {
            throw new IllegalArgumentException(String.format("The maximum number of tree lock nodes '%d' should be greater than the minimum number of tree lock nodes '%d'", maxNodesInMemory, minNodesInMemory));
        }
        this.maxTreeNodeInMemory = maxNodesInMemory;
        this.minTreeNodeInMemory = minNodesInMemory;
        long cleanIntervalInSecs = config.get(Configs.TREE_LOCK_CLEAN_INTERVAL);
        if (cleanIntervalInSecs <= 0L) {
            throw new IllegalArgumentException(String.format("The interval in seconds to clean up the stale tree lock nodes '%d' should be greater than 0", cleanIntervalInSecs));
        }
        this.cleanTreeNodeIntervalInSecs = cleanIntervalInSecs;
    }

    private void startDeadLockChecker() {
        ScheduledThreadPoolExecutor deadLockChecker = new ScheduledThreadPoolExecutor(1, new ThreadFactoryBuilder().setDaemon(true).setNameFormat("tree-lock-dead-lock-checker-%d").build());
        deadLockChecker.scheduleAtFixedRate(() -> {
            LOG.debug("Start to check the dead lock...");
            this.checkDeadLock(this.treeLockRootNode);
            LOG.debug("Finish to check the dead lock...");
        }, 0L, 60L, TimeUnit.SECONDS);
    }

    void checkDeadLock(TreeLockNode node) {
        node.getAllChildren().forEach(this::checkDeadLock);
        node.getHoldingThreadTimestamp().forEach((threadIdentifier, ts) -> {
            if (System.currentTimeMillis() - ts > 30000L) {
                LOG.warn("Thread with identifier {} holds the lock node {} for more than 30s since {}, please check if some dead lock or thread hang like io-connection hangs", new Object[]{threadIdentifier, node, new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(node.getHoldingThreadTimestamp())});
            }
        });
    }

    private void startNodeCleaner() {
        ScheduledThreadPoolExecutor lockCleaner = new ScheduledThreadPoolExecutor(1, new ThreadFactoryBuilder().setDaemon(true).setNameFormat("tree-lock-cleaner-%d").build());
        lockCleaner.scheduleAtFixedRate(() -> {
            long nodeCount = this.totalNodeCount.get();
            LOG.debug("Total tree lock node count: {}", (Object)nodeCount);
            if ((double)nodeCount > (double)this.maxTreeNodeInMemory * 0.5) {
                StopWatch watch = StopWatch.createStarted();
                LOG.info("Start to clean up the stale tree lock nodes...");
                this.treeLockRootNode.getAllChildren().forEach(child -> this.evictStaleNodes((TreeLockNode)child, this.treeLockRootNode));
                LOG.info("Finish to clean up the stale tree lock nodes, cost: {}, after clean node count: {}", (Object)watch.getTime(), (Object)this.totalNodeCount.get());
            }
        }, this.cleanTreeNodeIntervalInSecs, this.cleanTreeNodeIntervalInSecs, TimeUnit.SECONDS);
    }

    public LockManager(Config config) {
        this.treeLockRootNode = new TreeLockNode(ROOT.name());
        this.initParameters(config);
        this.startNodeCleaner();
        this.startDeadLockChecker();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    void evictStaleNodes(TreeLockNode treeNode, TreeLockNode parent) {
        if (this.totalNodeCount.get() < this.minTreeNodeInMemory) {
            return;
        }
        treeNode.getAllChildren().forEach(child -> this.evictStaleNodes((TreeLockNode)child, treeNode));
        if (treeNode.getReference() == 0L) {
            TreeLockNode treeLockNode = parent;
            synchronized (treeLockNode) {
                if (treeNode.getReference() == 0L) {
                    parent.removeChild(treeNode.getName());
                    long leftNodeCount = this.totalNodeCount.decrementAndGet();
                    if (LOG.isTraceEnabled()) {
                        LOG.trace("Evict stale tree lock node '{}', current left nodes '{}'", (Object)treeNode.getName(), (Object)leftNodeCount);
                    }
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public TreeLock createTreeLock(NameIdentifier identifier) {
        this.checkTreeNodeIsFull();
        ArrayList treeLockNodes = Lists.newArrayList();
        try {
            TreeLockNode lockNode = this.treeLockRootNode;
            lockNode.addReference();
            treeLockNodes.add(lockNode);
            if (identifier == ROOT) {
                return new TreeLock(treeLockNodes, identifier);
            }
            Object[] levels = identifier.namespace().levels();
            for (Object level : levels = (String[])ArrayUtils.add((Object[])levels, (Object)identifier.name())) {
                TreeLockNode child;
                TreeLockNode treeLockNode = lockNode;
                synchronized (treeLockNode) {
                    Pair<TreeLockNode, Boolean> pair = lockNode.getOrCreateChild((String)level);
                    child = (TreeLockNode)pair.getKey();
                    if (((Boolean)pair.getValue()).booleanValue()) {
                        this.totalNodeCount.incrementAndGet();
                    }
                }
                treeLockNodes.add(child);
                lockNode = child;
            }
            return new TreeLock(treeLockNodes, identifier);
        }
        catch (Exception e) {
            LOG.error("Failed to create tree lock {}", (Object)identifier, (Object)e);
            for (TreeLockNode node : treeLockNodes) {
                node.decReference();
            }
            throw e;
        }
    }

    private void checkTreeNodeIsFull() {
        long currentNodeCount = this.totalNodeCount.get();
        if (currentNodeCount > this.maxTreeNodeInMemory) {
            throw new IllegalStateException("The total node count '" + currentNodeCount + "' has reached the max node count '" + this.maxTreeNodeInMemory + "', please increase the max node count or wait for a while to avoid the performance issue.");
        }
    }
}

